/**
 * GRID TRADING API SERVER
 *
 * HTTP API for controlling Grid Trading Engine
 * Supports Demo, Live, and Backtest modes
 *
 * Port: 3037
 */

import http from 'http';
import { URL } from 'url';
import fs from 'fs';
import path from 'path';
import { GridTradingEngine, GRID_ENGINE_PRESETS, type GridEngineConfig } from './grid-trading-engine.js';
import { analyzeFromKlines, getStrategyParams, STRATEGY_PRESETS, type MarketAnalysis, type StrategyPreset } from './market-analyzer.js';
import dotenv from 'dotenv';

dotenv.config();

// ============= CONFIG PERSISTENCE =============

const CONFIG_DIR = path.join(process.cwd(), 'config');
const LIVE_CONFIG_FILE = path.join(CONFIG_DIR, 'live-trading-config.json');

// Ensure config directory exists
if (!fs.existsSync(CONFIG_DIR)) {
  fs.mkdirSync(CONFIG_DIR, { recursive: true });
}

interface SavedConfig {
  mode: string;
  symbols: string[];
  initialCapital: number;
  gridSpacing: number;
  gridLevels: number;
  positionPerGrid: number;
  leverage: number;
  maxDrawdownPercent: number;
  useLimitOrders: boolean;
  autoStrategy: boolean;
  bidirectional: boolean;
  autoBidirectional: boolean;
  savedAt: string;
  autoRestart: boolean;
}

function saveActiveConfig(config: Partial<GridEngineConfig>): void {
  try {
    const savedConfig: SavedConfig = {
      mode: config.mode || 'demo',
      symbols: config.symbols || ['BTC', 'ETH'],
      initialCapital: config.initialCapital || 1000,
      gridSpacing: config.gridSpacing || 0.4,
      gridLevels: config.gridLevels || 5,
      positionPerGrid: config.positionPerGrid || 25,
      leverage: config.leverage || 10,
      maxDrawdownPercent: config.maxDrawdownPercent || 20,
      useLimitOrders: config.useLimitOrders ?? true,
      autoStrategy: config.autoStrategy ?? false,
      bidirectional: config.bidirectional ?? true,
      autoBidirectional: config.autoBidirectional ?? false,
      savedAt: new Date().toISOString(),
      autoRestart: true
    };

    fs.writeFileSync(LIVE_CONFIG_FILE, JSON.stringify(savedConfig, null, 2));
    console.log(`[Config] Saved ${config.mode?.toUpperCase()} config to ${LIVE_CONFIG_FILE}`);
  } catch (error: any) {
    console.error(`[Config] Failed to save config: ${error.message}`);
  }
}

function loadSavedConfig(): SavedConfig | null {
  try {
    if (fs.existsSync(LIVE_CONFIG_FILE)) {
      const data = fs.readFileSync(LIVE_CONFIG_FILE, 'utf-8');
      const config = JSON.parse(data) as SavedConfig;
      console.log(`[Config] Loaded saved config from ${LIVE_CONFIG_FILE}`);
      console.log(`[Config] Mode: ${config.mode}, Symbols: ${config.symbols.join(',')}, Saved: ${config.savedAt}`);
      return config;
    }
  } catch (error: any) {
    console.error(`[Config] Failed to load config: ${error.message}`);
  }
  return null;
}

function clearSavedConfig(): void {
  try {
    if (fs.existsSync(LIVE_CONFIG_FILE)) {
      fs.unlinkSync(LIVE_CONFIG_FILE);
      console.log(`[Config] Cleared saved config`);
    }
  } catch (error: any) {
    console.error(`[Config] Failed to clear config: ${error.message}`);
  }
}

// ============= GLOBAL STATE =============

let gridEngine: GridTradingEngine | null = null;
let serverStartTime = Date.now();

// ============= UTILITY FUNCTIONS =============

function parseBody(req: http.IncomingMessage): Promise<any> {
  return new Promise((resolve, reject) => {
    let body = '';
    req.on('data', chunk => body += chunk);
    req.on('end', () => {
      try {
        resolve(body ? JSON.parse(body) : {});
      } catch (e) {
        reject(e);
      }
    });
    req.on('error', reject);
  });
}

function sendJson(res: http.ServerResponse, data: any, statusCode: number = 200): void {
  res.writeHead(statusCode, {
    'Content-Type': 'application/json',
    'Access-Control-Allow-Origin': '*',
    'Access-Control-Allow-Methods': 'GET, POST, PUT, DELETE, OPTIONS',
    'Access-Control-Allow-Headers': 'Content-Type'
  });
  res.end(JSON.stringify(data, null, 2));
}

function sendError(res: http.ServerResponse, message: string, statusCode: number = 400): void {
  sendJson(res, { success: false, error: message }, statusCode);
}

// ============= API HANDLERS =============

async function handleStatus(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendJson(res, {
      isRunning: false,
      mode: null,
      message: 'Grid engine not initialized',
      serverUptime: Math.floor((Date.now() - serverStartTime) / 1000)
    });
    return;
  }

  const status = gridEngine.getStatus();
  const tickers = gridEngine.getTickers();
  const config = gridEngine.getConfig();

  // Build symbol details
  const symbolDetails: any[] = [];
  for (const symbol of config.symbols) {
    const state = gridEngine.getSymbolState(symbol);
    const ticker = tickers.get(symbol);

    symbolDetails.push({
      symbol,
      price: ticker?.last || 0,
      change24h: ticker?.change24h || 0,
      basePrice: state?.basePrice || 0,
      equity: state?.equity || 0,
      realizedPnL: state?.realizedPnL || 0,
      unrealizedPnL: state?.unrealizedPnL || 0,
      positions: state?.positions.length || 0,
      gridLevels: state?.gridLevels.length || 0,
      trades: state?.trades.length || 0,
      maxDrawdown: state?.maxDrawdown || 0,
      isPaused: state?.isPaused || false,
      pauseReason: state?.pauseReason || null
    });
  }

  sendJson(res, {
    ...status,
    serverUptime: Math.floor((Date.now() - serverStartTime) / 1000),
    symbols: symbolDetails,
    config: {
      gridSpacing: config.gridSpacing,
      gridLevels: config.gridLevels,
      positionPerGrid: config.positionPerGrid,
      leverage: config.leverage,
      maxOpenPositions: config.maxOpenPositions,
      maxDrawdownPercent: config.maxDrawdownPercent,
      trendFilterEnabled: config.trendFilterEnabled,
      useLimitOrders: config.useLimitOrders,
      dynamicGrid: config.dynamicGrid,
      gridRebalanceThreshold: config.gridRebalanceThreshold
    }
  });
}

async function handleStart(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (gridEngine?.getStatus().isRunning) {
    sendError(res, 'Grid engine already running');
    return;
  }

  try {
    const body = await parseBody(req);

    // Get API keys based on mode
    const mode = body.mode || 'demo';
    let apiKey, apiSecret;

    if (mode === 'live') {
      apiKey = process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY;
      apiSecret = process.env.KRAKEN_FUTURES_API_SECRET || process.env.KRAKEN_API_SECRET;
    } else {
      apiKey = process.env.KRAKEN_DEMO_API_KEY;
      apiSecret = process.env.KRAKEN_DEMO_API_SECRET;
    }

    // Apply preset if specified
    let presetConfig: Partial<GridEngineConfig> = {};
    if (body.preset && GRID_ENGINE_PRESETS[body.preset as keyof typeof GRID_ENGINE_PRESETS]) {
      presetConfig = GRID_ENGINE_PRESETS[body.preset as keyof typeof GRID_ENGINE_PRESETS];
    }

    const config: Partial<GridEngineConfig> = {
      mode,
      symbols: body.symbols || ['BTC', 'ETH', 'SOL', 'XRP', 'LINK', 'ADA'],
      initialCapital: body.initialCapital || 3000,
      ...presetConfig,
      ...body.config,
      // Override with specific body parameters if provided
      gridSpacing: body.gridSpacing ?? presetConfig.gridSpacing ?? 0.5,
      gridLevels: body.gridLevels ?? presetConfig.gridLevels ?? 10,
      leverage: body.leverage ?? presetConfig.leverage ?? 5,
      maxDrawdownPercent: body.maxDrawdownPercent ?? presetConfig.maxDrawdownPercent ?? 30,
      positionPerGrid: body.positionPerGrid ?? 50,
      useLimitOrders: body.useLimitOrders ?? true,
      autoStrategy: body.autoStrategy || false,
      bidirectional: body.bidirectional !== false, // Default TRUE
      autoBidirectional: body.autoBidirectional || false, // Auto enable/disable SHORT based on trend
      immediateEntry: body.immediateEntry || false, // Open position immediately at start
      useSimulatedCapital: body.useSimulatedCapital || false, // Use initialCapital instead of real Kraken balance
      dynamicGrid: body.dynamicGrid || false, // Dynamic grid that follows price
      gridRebalanceThreshold: body.gridRebalanceThreshold || 2, // % deviation to trigger rebalance
      // HYPER-DYNAMIC GRID parameters
      hyperDynamic: body.hyperDynamic ?? false, // Enable ATR-based adaptive grid
      atrMultiplier: body.atrMultiplier ?? 0.5, // Grid spacing = ATR% × multiplier
      minGridSpacing: body.minGridSpacing ?? 0.15, // Minimum grid spacing %
      maxGridSpacing: body.maxGridSpacing ?? 1.5, // Maximum grid spacing %
      volatilityThreshold: body.volatilityThreshold ?? 0.3, // Min ATR% to trade (0 = disabled)
      tickRecalibration: body.tickRecalibration !== false, // Recalibrate on every tick
      apiKey,
      apiSecret
    };

    gridEngine = new GridTradingEngine(config);

    // Setup event listeners
    gridEngine.on('started', (data) => console.log('[API] Grid engine started:', data));
    gridEngine.on('stopped', () => console.log('[API] Grid engine stopped'));
    gridEngine.on('position_opened', (pos) => console.log(`[API] Position opened: ${pos.symbol} @ ${pos.entryPrice}`));
    gridEngine.on('position_closed', (trade) => console.log(`[API] Position closed: ${trade.symbol} PnL=${trade.netPnl.toFixed(2)}`));
    gridEngine.on('circuit_breaker', (data) => console.log(`[API] Circuit breaker: ${data.symbol} DD=${data.drawdown.toFixed(1)}%`));
    gridEngine.on('strategy_changed', (data) => console.log(`[API] Strategy changed: ${data.from || 'none'} -> ${data.to} (${data.reason})`));
    gridEngine.on('bidirectional_change', (data) => console.log(`[API] Bidirectional change: ${data.symbol} - ${data.reason}`));
    gridEngine.on('grid_rebalanced', (data) => console.log(`[API] Dynamic Grid Rebalanced: ${data.symbol} ${data.oldBase.toFixed(2)} -> ${data.newBase.toFixed(2)} (${data.deviation.toFixed(2)}%)`));
    gridEngine.on('hyper_recalibration', (data) => console.log(`[API] HyperDynamic Recalibration: ${data.symbol} spacing ${data.oldSpacing.toFixed(3)}% -> ${data.newSpacing.toFixed(3)}% (ATR: ${data.atrPercent.toFixed(3)}%)`));
    gridEngine.on('trading_paused', (data) => console.log(`[API] Trading PAUSED: ${data.symbol} - ${data.reason}`));
    gridEngine.on('trading_resumed', (data) => console.log(`[API] Trading RESUMED: ${data.symbol}`));

    await gridEngine.start();

    // Return config without sensitive data
    const fullConfig = gridEngine.getConfig();
    const { apiKey: _ak, apiSecret: _as, ...safeConfig } = fullConfig;

    // Save active config for persistence (only for live and demo modes)
    if (mode === 'live' || mode === 'demo') {
      saveActiveConfig(fullConfig);
    }

    sendJson(res, {
      success: true,
      message: `Grid engine started in ${mode} mode`,
      config: safeConfig
    });
  } catch (error: any) {
    sendError(res, `Failed to start: ${error.message}`, 500);
  }
}

async function handleStop(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendError(res, 'Grid engine not initialized');
    return;
  }

  try {
    const body = await parseBody(req);
    const status = gridEngine.getStatus();
    const mode = status.mode || 'demo';

    // Close all positions if requested
    if (body.closePositions) {
      await gridEngine.closeAllPositions();
    }

    // Cancel all open orders on Kraken
    const apiKey = mode === 'live'
      ? (process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY)
      : process.env.KRAKEN_DEMO_API_KEY;
    const apiSecret = mode === 'live'
      ? (process.env.KRAKEN_FUTURES_API_SECRET || process.env.KRAKEN_API_SECRET)
      : process.env.KRAKEN_DEMO_API_SECRET;

    let cancelledOrders = 0;
    if (apiKey && apiSecret) {
      console.log(`[GridAPI] Cancelling all open orders on Kraken (${mode} mode)...`);
      cancelledOrders = await cancelAllKrakenOrdersWithCount(mode, apiKey, apiSecret);
      console.log(`[GridAPI] Cancelled ${cancelledOrders} orders`);
    }

    gridEngine.stop();
    gridEngine = null;

    // Clear saved config on explicit stop (unless keepConfig is true)
    if (!body.keepConfig) {
      clearSavedConfig();
    }

    sendJson(res, {
      success: true,
      message: `Grid engine stopped. Cancelled ${cancelledOrders} orders on Kraken.`
    });
  } catch (error: any) {
    sendError(res, `Failed to stop: ${error.message}`, 500);
  }
}

async function handlePause(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendError(res, 'Grid engine not initialized');
    return;
  }

  const body = await parseBody(req);
  gridEngine.pause(body.symbol);

  sendJson(res, {
    success: true,
    message: `Paused ${body.symbol || 'all symbols'}`
  });
}

async function handleResume(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendError(res, 'Grid engine not initialized');
    return;
  }

  const body = await parseBody(req);
  gridEngine.resume(body.symbol);

  sendJson(res, {
    success: true,
    message: `Resumed ${body.symbol || 'all symbols'}`
  });
}

async function handleRebalance(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendError(res, 'Grid engine not initialized');
    return;
  }

  gridEngine.rebalanceGrids();

  sendJson(res, {
    success: true,
    message: 'Grids rebalanced to current prices'
  });
}

async function handleCloseAll(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendError(res, 'Grid engine not initialized');
    return;
  }

  await gridEngine.closeAllPositions();

  sendJson(res, {
    success: true,
    message: 'All positions closed'
  });
}

async function handlePositions(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendJson(res, { positions: [] });
    return;
  }

  const positions: any[] = [];
  const config = gridEngine.getConfig();

  for (const symbol of config.symbols) {
    const state = gridEngine.getSymbolState(symbol);
    if (state) {
      for (const pos of state.positions) {
        positions.push({
          ...pos,
          symbol,
          currentPrice: state.currentPrice,
          unrealizedPnL: (state.currentPrice - pos.entryPrice) * pos.size,
          unrealizedPnLPercent: ((state.currentPrice - pos.entryPrice) / pos.entryPrice) * 100
        });
      }
    }
  }

  sendJson(res, { positions });
}

async function handleOpenOrders(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  const crypto = await import('crypto');

  // Determine mode from grid engine or default to demo
  const mode = gridEngine?.getStatus()?.mode || 'demo';
  const isDemo = mode !== 'live';

  const apiKey = isDemo ? process.env.KRAKEN_DEMO_API_KEY : process.env.KRAKEN_FUTURES_API_KEY;
  const apiSecret = isDemo ? process.env.KRAKEN_DEMO_API_SECRET : process.env.KRAKEN_FUTURES_API_SECRET;
  const baseUrl = isDemo
    ? 'https://demo-futures.kraken.com/derivatives/api/v3'
    : 'https://futures.kraken.com/derivatives/api/v3';

  if (!apiKey || !apiSecret) {
    sendJson(res, { orders: [], error: 'API keys not configured' });
    return;
  }

  try {
    const endpoint = '/openorders';
    const fullEndpoint = '/api/v3' + endpoint;
    const nonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
    const postData = '';

    const message = postData + nonce + fullEndpoint;
    const hash = crypto.createHash('sha256').update(message).digest();
    const signature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64'))
      .update(hash)
      .digest('base64');

    const response = await fetch(`${baseUrl}${endpoint}`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'APIKey': apiKey,
        'Nonce': nonce,
        'Authent': signature,
      }
    });

    const data = await response.json() as any;

    if (data.result === 'success' && data.openOrders) {
      // Format orders for UI
      const orders = data.openOrders.map((o: any) => ({
        id: o.order_id,
        symbol: o.symbol,
        side: o.side,
        size: o.quantity || o.unfilledSize,
        price: o.limitPrice,
        filled: o.filledSize || 0,
        status: o.status,
        timestamp: o.lastUpdateTime || o.receivedTime
      }));

      sendJson(res, {
        success: true,
        orders,
        count: orders.length,
        mode: isDemo ? 'demo' : 'live'
      });
    } else {
      sendJson(res, { orders: [], error: data.error || 'Failed to fetch orders' });
    }
  } catch (error: any) {
    sendJson(res, { orders: [], error: error.message });
  }
}

async function handleTrades(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendJson(res, { trades: [] });
    return;
  }

  const trades: any[] = [];
  const config = gridEngine.getConfig();

  for (const symbol of config.symbols) {
    const state = gridEngine.getSymbolState(symbol);
    if (state) {
      trades.push(...state.trades);
    }
  }

  // Sort by exit time descending
  trades.sort((a, b) => new Date(b.exitTime).getTime() - new Date(a.exitTime).getTime());

  sendJson(res, { trades: trades.slice(0, 100) }); // Last 100 trades
}

async function handleGridLevels(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (!gridEngine) {
    sendJson(res, { levels: {} });
    return;
  }

  const levels: Record<string, any[]> = {};
  const config = gridEngine.getConfig();

  for (const symbol of config.symbols) {
    const state = gridEngine.getSymbolState(symbol);
    if (state) {
      levels[symbol] = state.gridLevels.map(g => ({
        ...g,
        distanceFromCurrent: state.currentPrice > 0
          ? ((g.price - state.currentPrice) / state.currentPrice * 100).toFixed(2) + '%'
          : '0%'
      }));
    }
  }

  sendJson(res, { levels });
}

async function handleConfig(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (req.method === 'GET') {
    if (!gridEngine) {
      sendJson(res, { config: null, presets: GRID_ENGINE_PRESETS });
      return;
    }
    sendJson(res, {
      config: gridEngine.getConfig(),
      presets: GRID_ENGINE_PRESETS
    });
    return;
  }

  if (req.method === 'PUT') {
    if (!gridEngine) {
      sendError(res, 'Grid engine not initialized');
      return;
    }

    const body = await parseBody(req);
    gridEngine.updateConfig(body);

    sendJson(res, {
      success: true,
      message: 'Config updated',
      config: gridEngine.getConfig()
    });
  }
}

async function handleTestConnection(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  const crypto = await import('crypto');

  async function testKrakenConnection(mode: 'live' | 'demo'): Promise<{ success: boolean; balance?: number; currency?: string; error?: string }> {
    const apiKey = mode === 'live'
      ? (process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY)
      : process.env.KRAKEN_DEMO_API_KEY;
    const apiSecret = mode === 'live'
      ? (process.env.KRAKEN_FUTURES_API_SECRET || process.env.KRAKEN_API_SECRET)
      : process.env.KRAKEN_DEMO_API_SECRET;

    if (!apiKey || !apiSecret) {
      return { success: false, error: `API keys not configured for ${mode} mode` };
    }

    const baseUrl = mode === 'live'
      ? 'https://futures.kraken.com/derivatives/api/v3'
      : 'https://demo-futures.kraken.com/derivatives/api/v3';

    const endpoint = '/accounts';
    const fullEndpoint = '/api/v3' + endpoint;  // Full path for signature
    // Use large nonce to avoid "nonceBelowThreshold" error
    const nonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
    const postData = '';

    // Correct signature per Kraken Futures docs:
    // Step 1: SHA256(postData + nonce + fullEndpoint)
    // Step 2: HMAC-SHA512(hash) with base64-decoded secret
    // Step 3: Base64-encode result
    const message = postData + nonce + fullEndpoint;
    const hash = crypto.createHash('sha256').update(message).digest();
    const signature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64'))
      .update(hash)
      .digest('base64');

    try {
      const response = await fetch(`${baseUrl}${endpoint}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
          'APIKey': apiKey,
          'Nonce': nonce,
          'Authent': signature,
        }
      });

      const data = await response.json() as any;

      if (data.result === 'success' && data.accounts) {
        // Priority: flex (multi-collateral) > cash > individual margin accounts
        if (data.accounts.flex) {
          const flex = data.accounts.flex;
          return {
            success: true,
            balance: flex.portfolioValue || flex.balanceValue || 0,
            availableMargin: flex.availableMargin || 0,
            currency: 'USD',
            currencies: flex.currencies  // Detailed breakdown
          };
        }

        if (data.accounts.cash) {
          const cash = data.accounts.cash;
          const usdBalance = cash.balances?.usd || cash.balances?.usdt || cash.balances?.usdc || 0;
          return {
            success: true,
            balance: usdBalance,
            currency: 'USD'
          };
        }

        // Fallback to first margin account
        const accountKeys = Object.keys(data.accounts);
        if (accountKeys.length > 0) {
          const firstAccount = data.accounts[accountKeys[0]];
          return {
            success: true,
            balance: firstAccount.auxiliary?.pv || 0,
            currency: firstAccount.currency || accountKeys[0]
          };
        }

        return { success: true, balance: 0, currency: 'USD' };
      }

      return { success: false, error: data.error || 'Unknown error' };
    } catch (error: any) {
      return { success: false, error: error.message };
    }
  }

  try {
    const [liveResult, demoResult] = await Promise.all([
      testKrakenConnection('live'),
      testKrakenConnection('demo')
    ]);

    sendJson(res, {
      success: true,
      live: {
        connected: liveResult.success,
        portfolioValue: liveResult.balance,
        availableMargin: (liveResult as any).availableMargin,
        currencies: (liveResult as any).currencies,
        currency: liveResult.currency,
        error: liveResult.error,
        apiKeyConfigured: !!(process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY),
        endpoint: 'futures.kraken.com'
      },
      demo: {
        connected: demoResult.success,
        portfolioValue: demoResult.balance,
        availableMargin: (demoResult as any).availableMargin,
        currencies: (demoResult as any).currencies,
        currency: demoResult.currency,
        error: demoResult.error,
        apiKeyConfigured: !!process.env.KRAKEN_DEMO_API_KEY,
        endpoint: 'demo-futures.kraken.com'
      },
      timestamp: new Date().toISOString()
    });
  } catch (error: any) {
    sendError(res, `Connection test failed: ${error.message}`, 500);
  }
}

// ============= DIRECT KRAKEN POSITIONS QUERY =============

async function handleKrakenPositions(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  const crypto = await import('crypto');
  const url = new URL(req.url || '/', `http://${req.headers.host}`);
  const mode = url.searchParams.get('mode') || 'live';

  const apiKey = mode === 'live'
    ? (process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY)
    : process.env.KRAKEN_DEMO_API_KEY;
  const apiSecret = mode === 'live'
    ? (process.env.KRAKEN_FUTURES_API_SECRET || process.env.KRAKEN_API_SECRET)
    : process.env.KRAKEN_DEMO_API_SECRET;

  if (!apiKey || !apiSecret) {
    sendError(res, `API keys not configured for ${mode} mode`, 400);
    return;
  }

  const baseUrl = mode === 'live'
    ? 'https://futures.kraken.com/derivatives/api/v3'
    : 'https://demo-futures.kraken.com/derivatives/api/v3';

  async function queryKraken(endpoint: string): Promise<any> {
    const fullEndpoint = '/api/v3' + endpoint;
    const nonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
    const postData = '';
    const message = postData + nonce + fullEndpoint;
    const hash = crypto.createHash('sha256').update(message).digest();
    const signature = crypto.createHmac('sha512', Buffer.from(apiSecret!, 'base64'))
      .update(hash)
      .digest('base64');

    const response = await fetch(`${baseUrl}${endpoint}`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'APIKey': apiKey!,
        'Nonce': nonce,
        'Authent': signature,
      }
    });
    return response.json();
  }

  try {
    const [positionsData, ordersData] = await Promise.all([
      queryKraken('/openpositions'),
      queryKraken('/openorders')
    ]);

    sendJson(res, {
      success: true,
      mode,
      positions: positionsData.openPositions || [],
      positionsCount: (positionsData.openPositions || []).length,
      orders: ordersData.openOrders || [],
      ordersCount: (ordersData.openOrders || []).length,
      timestamp: new Date().toISOString()
    });
  } catch (error: any) {
    sendError(res, `Kraken query failed: ${error.message}`, 500);
  }
}

async function handlePresets(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  sendJson(res, {
    presets: GRID_ENGINE_PRESETS,
    description: {
      conservative: 'Low risk, stable returns. Good for beginners.',
      balanced: 'Medium risk/reward. Recommended for most users.',
      aggressive: 'High risk, high potential returns. For experienced traders.'
    }
  });
}

async function handleSavedConfig(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (req.method === 'GET') {
    const savedConfig = loadSavedConfig();
    sendJson(res, {
      hasSavedConfig: !!savedConfig,
      config: savedConfig,
      configFile: LIVE_CONFIG_FILE
    });
    return;
  }

  if (req.method === 'DELETE') {
    clearSavedConfig();
    sendJson(res, {
      success: true,
      message: 'Saved config cleared'
    });
    return;
  }

  sendError(res, 'Method not allowed', 405);
}

async function handleHealth(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  sendJson(res, {
    status: 'ok',
    timestamp: Date.now(),
    uptime: Math.floor((Date.now() - serverStartTime) / 1000),
    gridEngineRunning: gridEngine?.getStatus().isRunning || false
  });
}

// ============= GRID BACKTESTER =============

interface BacktestConfig {
  symbol: string;
  startDate: string;      // YYYY-MM-DD
  endDate: string;
  initialCapital: number;
  gridSpacing: number;    // % between levels
  gridLevels: number;     // levels above/below
  positionSize: number;   // $ per position
  leverage: number;
  takeProfitPercent: number;
  autoStrategy?: boolean; // Enable automatic strategy selection
  bidirectional?: boolean; // Enable both LONG and SHORT positions
}

interface StrategyChange {
  time: number;
  fromStrategy: StrategyPreset | null;
  toStrategy: StrategyPreset;
  reason: string;
  marketCondition: string;
  confidence: number;
}

interface BacktestTrade {
  entryTime: number;
  exitTime: number;
  entryPrice: number;
  exitPrice: number;
  side: 'LONG' | 'SHORT';
  size: number;
  pnl: number;
  fees: number;
  netPnl: number;
  roe: number;
}

interface BacktestResult {
  symbol: string;
  period: { start: string; end: string };
  config: BacktestConfig;
  summary: {
    totalTrades: number;
    winningTrades: number;
    losingTrades: number;
    winRate: number;
    totalPnL: number;
    totalFees: number;
    netPnL: number;
    maxDrawdown: number;
    maxDrawdownPercent: number;
    sharpeRatio: number;
    profitFactor: number;
    avgTradeProfit: number;
    avgWinProfit: number;
    avgLossProfit: number;
    maxConsecutiveWins: number;
    maxConsecutiveLosses: number;
  };
  trades: BacktestTrade[];
  equityCurve: { time: number; equity: number }[];
  priceCurve: { time: number; price: number }[];  // BTC/ETH price for comparison
  monthlyReturns: { month: string; pnl: number; trades: number }[];
  strategyChanges?: StrategyChange[];
  autoStrategyUsed?: boolean;
}

// Fetch real tick data (aggregated trades) from Binance
async function* fetchBinanceTicks(symbol: string, startTime: number, endTime: number): AsyncGenerator<{ time: number; price: number; qty: number; isBuy: boolean }> {
  let fromId: number | null = null;
  let currentTime = startTime;
  const limit = 1000;

  console.log(`[Backtest-Tick] Fetching ${symbol}USDT ticks from ${new Date(startTime).toISOString()}`);

  // Get starting trade ID
  const startUrl = `https://fapi.binance.com/fapi/v1/aggTrades?symbol=${symbol}USDT&startTime=${startTime}&limit=1`;
  try {
    const startResp = await fetch(startUrl);
    if (startResp.ok) {
      const startData = await startResp.json() as any[];
      if (startData.length > 0) {
        fromId = startData[0].a;
      }
    }
  } catch (e) {
    console.error('[Backtest-Tick] Error getting start trade:', e);
  }

  let tickCount = 0;

  while (currentTime < endTime) {
    let url: string;
    if (fromId !== null) {
      url = `https://fapi.binance.com/fapi/v1/aggTrades?symbol=${symbol}USDT&fromId=${fromId}&limit=${limit}`;
    } else {
      url = `https://fapi.binance.com/fapi/v1/aggTrades?symbol=${symbol}USDT&startTime=${currentTime}&endTime=${endTime}&limit=${limit}`;
    }

    try {
      const response = await fetch(url);
      if (!response.ok) {
        if (response.status === 429) {
          await new Promise(r => setTimeout(r, 5000));
          continue;
        }
        throw new Error(`Binance API error: ${response.status}`);
      }

      const trades = await response.json() as any[];
      if (trades.length === 0) break;

      for (const t of trades) {
        if (t.T > endTime) return;

        yield {
          time: t.T,
          price: parseFloat(t.p),
          qty: parseFloat(t.q),
          isBuy: !t.m  // m = true means seller is maker, so buyer is taker
        };
        tickCount++;
      }

      fromId = trades[trades.length - 1].a + 1;
      currentTime = trades[trades.length - 1].T;

      // Rate limiting
      await new Promise(r => setTimeout(r, 50));

      // Log progress every 10k ticks
      if (tickCount % 10000 === 0) {
        console.log(`[Backtest-Tick] ${tickCount} ticks processed, at ${new Date(currentTime).toISOString()}`);
      }

    } catch (error) {
      console.error('[Backtest-Tick] Fetch error:', error);
      break;
    }
  }

  console.log(`[Backtest-Tick] Total: ${tickCount} ticks`);
}

// Run TICK-BY-TICK backtest (more accurate but slower)
async function runGridBacktestTickByTick(config: BacktestConfig): Promise<BacktestResult> {
  const startTime = new Date(config.startDate).getTime();
  const endTime = new Date(config.endDate).getTime();

  // Backtest state
  let equity = config.initialCapital;
  let maxEquity = equity;
  let maxDrawdown = 0;
  let maxDrawdownPercent = 0;

  const trades: BacktestTrade[] = [];
  const equityCurve: { time: number; equity: number }[] = [];
  const priceCurve: { time: number; price: number }[] = [];

  // Grid state
  let basePrice = 0;
  let gridSpacing = config.gridSpacing;
  let gridLevels_count = config.gridLevels;
  let gridLevels: { price: number; level: number; filled: boolean }[] = [];

  interface Position {
    entryPrice: number;
    size: number;
    entryTime: number;
    level: number;
    tpPrice: number;
    leverage: number;
  }
  const positions: Position[] = [];

  const MAKER_FEE = 0.0002;
  const TAKER_FEE = 0.0005;

  const rebuildGrid = (price: number) => {
    basePrice = price;
    gridLevels = [];
    for (let l = 1; l <= gridLevels_count; l++) {
      const levelPrice = basePrice * (1 - (gridSpacing / 100) * l);
      gridLevels.push({ price: levelPrice, level: -l, filled: false });
    }
  };

  let tickCount = 0;
  let lastEquityLog = startTime;

  // Process each tick
  for await (const tick of fetchBinanceTicks(config.symbol, startTime, endTime)) {
    tickCount++;
    const price = tick.price;

    // Initialize grid on first tick
    if (basePrice === 0) {
      rebuildGrid(price);
      equityCurve.push({ time: tick.time, equity });
      priceCurve.push({ time: tick.time, price });
    }

    // Check if any grid level was hit
    for (const level of gridLevels) {
      if (!level.filled && price <= level.price) {
        // Grid level hit - open position
        const posValue = config.positionSize;
        const size = (posValue * config.leverage) / level.price;
        const entryFee = posValue * config.leverage * TAKER_FEE;

        const tpPrice = level.price * (1 + gridSpacing / 100);

        positions.push({
          entryPrice: level.price,
          size,
          entryTime: tick.time,
          level: level.level,
          tpPrice,
          leverage: config.leverage
        });

        level.filled = true;
        equity -= entryFee;

        console.log(`[Tick] ENTRY at ${level.price.toFixed(2)} (level ${level.level}), TP: ${tpPrice.toFixed(2)}`);
      }
    }

    // Check take profits
    for (let p = positions.length - 1; p >= 0; p--) {
      const pos = positions[p];

      if (price >= pos.tpPrice) {
        const pnl = (pos.tpPrice - pos.entryPrice) * pos.size;
        const posValue = (pos.entryPrice * pos.size) / pos.leverage;
        const entryFee = posValue * pos.leverage * TAKER_FEE;
        const exitFee = (pos.size * pos.tpPrice) * MAKER_FEE;
        const totalFees = entryFee + exitFee;
        const netPnl = pnl - totalFees;
        const roe = (netPnl / posValue) * 100;

        trades.push({
          entryTime: pos.entryTime,
          exitTime: tick.time,
          entryPrice: pos.entryPrice,
          exitPrice: pos.tpPrice,
          side: 'LONG',
          size: pos.size,
          pnl,
          fees: totalFees,
          netPnl,
          roe
        });

        equity += netPnl;

        // Reset grid level
        const level = gridLevels.find(l => l.level === pos.level);
        if (level) level.filled = false;

        positions.splice(p, 1);

        console.log(`[Tick] EXIT at ${pos.tpPrice.toFixed(2)}, PnL: $${netPnl.toFixed(4)}`);
      }
    }

    // Calculate unrealized PnL
    let unrealizedPnL = 0;
    for (const pos of positions) {
      unrealizedPnL += (price - pos.entryPrice) * pos.size;
    }

    const totalEquity = equity + unrealizedPnL;

    // Track drawdown
    if (totalEquity > maxEquity) maxEquity = totalEquity;
    const drawdown = maxEquity - totalEquity;
    if (drawdown > maxDrawdown) {
      maxDrawdown = drawdown;
      maxDrawdownPercent = (drawdown / maxEquity) * 100;
    }

    // Record equity every 5 minutes
    if (tick.time - lastEquityLog >= 5 * 60 * 1000) {
      equityCurve.push({ time: tick.time, equity: totalEquity });
      priceCurve.push({ time: tick.time, price });
      lastEquityLog = tick.time;
    }
  }

  // Close remaining positions at last price
  const lastTick = priceCurve[priceCurve.length - 1];
  const lastPrice = lastTick?.price || basePrice;

  for (const pos of positions) {
    const pnl = (lastPrice - pos.entryPrice) * pos.size;
    const entryFee = config.positionSize * config.leverage * TAKER_FEE;
    const exitFee = (pos.size * lastPrice) * TAKER_FEE;
    const totalFees = entryFee + exitFee;
    const netPnl = pnl - totalFees;
    const roe = (netPnl / config.positionSize) * 100;

    trades.push({
      entryTime: pos.entryTime,
      exitTime: lastTick?.time || Date.now(),
      entryPrice: pos.entryPrice,
      exitPrice: lastPrice,
      side: 'LONG',
      size: pos.size,
      pnl,
      fees: totalFees,
      netPnl,
      roe
    });

    equity += netPnl;
  }

  // Calculate summary stats
  const winningTrades = trades.filter(t => t.netPnl > 0);
  const losingTrades = trades.filter(t => t.netPnl <= 0);
  const totalPnL = trades.reduce((sum, t) => sum + t.pnl, 0);
  const totalFees = trades.reduce((sum, t) => sum + t.fees, 0);
  const netPnL = trades.reduce((sum, t) => sum + t.netPnl, 0);
  const grossProfit = winningTrades.reduce((sum, t) => sum + t.netPnl, 0);
  const grossLoss = Math.abs(losingTrades.reduce((sum, t) => sum + t.netPnl, 0));
  const profitFactor = grossLoss > 0 ? grossProfit / grossLoss : grossProfit > 0 ? Infinity : 0;

  console.log(`[Backtest-Tick] Complete: ${tickCount} ticks, ${trades.length} trades`);

  return {
    symbol: config.symbol,
    period: { start: config.startDate, end: config.endDate },
    config,
    summary: {
      totalTrades: trades.length,
      winningTrades: winningTrades.length,
      losingTrades: losingTrades.length,
      winRate: trades.length > 0 ? (winningTrades.length / trades.length) * 100 : 0,
      totalPnL,
      totalFees,
      netPnL,
      maxDrawdown,
      maxDrawdownPercent,
      sharpeRatio: 0,
      profitFactor,
      avgTradeProfit: trades.length > 0 ? netPnL / trades.length : 0,
      avgWinProfit: winningTrades.length > 0 ? grossProfit / winningTrades.length : 0,
      avgLossProfit: losingTrades.length > 0 ? -grossLoss / losingTrades.length : 0,
      maxConsecutiveWins: 0,
      maxConsecutiveLosses: 0
    },
    trades,
    equityCurve,
    priceCurve,
    monthlyReturns: [],
    autoStrategyUsed: false
  };
}

async function fetchBinanceKlines(symbol: string, startTime: number, endTime: number, interval: string = '1m'): Promise<any[]> {
  const klines: any[] = [];
  let currentStart = startTime;
  const limit = 1000;

  console.log(`[Backtest] Fetching ${symbol} data from ${new Date(startTime).toISOString()} to ${new Date(endTime).toISOString()}`);

  while (currentStart < endTime) {
    const url = `https://fapi.binance.com/fapi/v1/klines?symbol=${symbol}USDT&interval=${interval}&startTime=${currentStart}&endTime=${endTime}&limit=${limit}`;

    try {
      const response = await fetch(url);
      if (!response.ok) {
        if (response.status === 429) {
          await new Promise(r => setTimeout(r, 5000));
          continue;
        }
        throw new Error(`Binance API error: ${response.status}`);
      }

      const data = await response.json() as any[];
      if (data.length === 0) break;

      klines.push(...data);
      currentStart = data[data.length - 1][0] + 1;

      // Rate limiting
      await new Promise(r => setTimeout(r, 100));
    } catch (error) {
      console.error('[Backtest] Fetch error:', error);
      break;
    }
  }

  console.log(`[Backtest] Fetched ${klines.length} candles`);
  return klines;
}

async function runGridBacktest(config: BacktestConfig): Promise<BacktestResult> {
  const startTime = new Date(config.startDate).getTime();
  const endTime = new Date(config.endDate).getTime();

  // Fetch historical data
  const klines = await fetchBinanceKlines(config.symbol, startTime, endTime, '1m');

  if (klines.length === 0) {
    throw new Error('No historical data available');
  }

  // Backtest state
  let equity = config.initialCapital;
  let maxEquity = equity;
  let maxDrawdown = 0;
  let maxDrawdownPercent = 0;

  const trades: BacktestTrade[] = [];
  const equityCurve: { time: number; equity: number }[] = [];
  const priceCurve: { time: number; price: number }[] = [];
  const strategyChanges: StrategyChange[] = [];
  const dailyReturns: number[] = [];

  // Dynamic strategy parameters (can change if autoStrategy enabled)
  let currentStrategy: StrategyPreset | null = null;
  let gridSpacing = config.gridSpacing;
  let gridLevels_count = config.gridLevels;
  let leverage = config.leverage;
  let positionSize = config.positionSize;

  // Grid state
  let basePrice = 0;
  // Grid levels now include both LONG (below) and SHORT (above) levels
  let gridLevels: { price: number; level: number; filled: boolean; side: 'LONG' | 'SHORT' }[] = [];
  interface Position {
    entryPrice: number;
    size: number;
    entryTime: number;
    level: number;
    tpPrice: number;
    leverage: number;
    side: 'LONG' | 'SHORT';
  }
  const positions: Position[] = [];

  const MAKER_FEE = 0.0002;  // 0.02%
  const TAKER_FEE = 0.0005;  // 0.05%

  // Auto strategy analysis interval (240 candles = 4 hours)
  const ANALYSIS_INTERVAL = 240;

  // Helper to rebuild grid - WITH OPTIONAL BIDIRECTIONAL (LONG + SHORT)
  const rebuildGrid = (price: number) => {
    basePrice = price;
    gridLevels = [];
    // LONG levels (below base price) - buy when price drops, sell when rises
    for (let l = 1; l <= gridLevels_count; l++) {
      const levelPrice = basePrice * (1 - (gridSpacing / 100) * l);
      gridLevels.push({ price: levelPrice, level: -l, filled: false, side: 'LONG' });
    }
    // SHORT levels (above base price) - only if bidirectional enabled
    if (config.bidirectional) {
      for (let l = 1; l <= gridLevels_count; l++) {
        const levelPrice = basePrice * (1 + (gridSpacing / 100) * l);
        gridLevels.push({ price: levelPrice, level: l, filled: false, side: 'SHORT' });
      }
    }
  };

  // Process each candle tick-by-tick
  for (let i = 0; i < klines.length; i++) {
    const [openTime, open, high, low, close] = klines[i];
    const currentPrice = parseFloat(close);
    const highPrice = parseFloat(high);
    const lowPrice = parseFloat(low);
    const timestamp = openTime;

    // === AUTO STRATEGY: Analyze market every ANALYSIS_INTERVAL candles ===
    if (config.autoStrategy && i % ANALYSIS_INTERVAL === 0 && i >= 100) {
      // Need at least 100 candles for analysis
      const analysisKlines = klines.slice(Math.max(0, i - 100), i + 1);
      const analysis = analyzeFromKlines(analysisKlines);

      const newStrategy = analysis.recommendation;

      // Only change if strategy is different
      if (newStrategy !== currentStrategy) {
        const params = getStrategyParams(newStrategy, equity);

        // Log strategy change
        strategyChanges.push({
          time: timestamp,
          fromStrategy: currentStrategy,
          toStrategy: newStrategy,
          reason: analysis.reason,
          marketCondition: analysis.condition,
          confidence: analysis.confidence
        });

        console.log(`[Backtest] Strategy change at ${new Date(timestamp).toISOString()}: ${currentStrategy || 'initial'} → ${newStrategy} (${analysis.condition})`);

        // Update parameters
        currentStrategy = newStrategy;
        gridSpacing = params.gridSpacing;
        gridLevels_count = params.gridLevels;
        leverage = params.leverage;
        positionSize = (equity * params.positionSizePercent) / 100;

        // Rebuild grid with new parameters (if we have a base price)
        if (basePrice > 0) {
          rebuildGrid(currentPrice);
        }
      }
    }

    // Initialize grid on first candle
    if (basePrice === 0) {
      // If auto strategy, do initial analysis
      if (config.autoStrategy && i >= 50) {
        const analysisKlines = klines.slice(0, i + 1);
        const analysis = analyzeFromKlines(analysisKlines);
        const params = getStrategyParams(analysis.recommendation, equity);

        currentStrategy = analysis.recommendation;
        gridSpacing = params.gridSpacing;
        gridLevels_count = params.gridLevels;
        leverage = params.leverage;
        positionSize = (equity * params.positionSizePercent) / 100;

        strategyChanges.push({
          time: timestamp,
          fromStrategy: null,
          toStrategy: currentStrategy,
          reason: `Initial: ${analysis.reason}`,
          marketCondition: analysis.condition,
          confidence: analysis.confidence
        });
      }

      rebuildGrid(currentPrice);
    }

    // Check if any grid level was hit (using high/low for tick simulation)
    for (const level of gridLevels) {
      if (!level.filled && lowPrice <= level.price && level.price <= highPrice) {
        // Grid level hit - open position
        const size = (positionSize * leverage) / level.price;
        const entryFee = positionSize * leverage * MAKER_FEE;

        // TP depends on side: LONG profits when price rises, SHORT profits when price drops
        const tpPrice = level.side === 'LONG'
          ? level.price * (1 + gridSpacing / 100)  // LONG: TP above entry
          : level.price * (1 - gridSpacing / 100); // SHORT: TP below entry

        positions.push({
          entryPrice: level.price,
          size,
          entryTime: timestamp,
          level: level.level,
          tpPrice,
          leverage,
          side: level.side
        });

        level.filled = true;
        equity -= entryFee;
      }
    }

    // Check take profits
    for (let p = positions.length - 1; p >= 0; p--) {
      const pos = positions[p];

      // Check if TP hit - depends on position side
      const tpHit = pos.side === 'LONG'
        ? highPrice >= pos.tpPrice   // LONG: price rises to TP
        : lowPrice <= pos.tpPrice;   // SHORT: price drops to TP

      if (tpHit) {
        // PnL calculation depends on side
        const pnl = pos.side === 'LONG'
          ? (pos.tpPrice - pos.entryPrice) * pos.size  // LONG: profit when price rises
          : (pos.entryPrice - pos.tpPrice) * pos.size; // SHORT: profit when price drops

        // TP uses limit order = MAKER_FEE
        const posValue = (pos.entryPrice * pos.size) / pos.leverage;
        const entryFee = posValue * pos.leverage * MAKER_FEE;
        const exitFee = (pos.size * pos.tpPrice) * MAKER_FEE;
        const totalFees = entryFee + exitFee;
        const netPnl = pnl - totalFees;
        const roe = (netPnl / posValue) * 100;

        trades.push({
          entryTime: pos.entryTime,
          exitTime: timestamp,
          entryPrice: pos.entryPrice,
          exitPrice: pos.tpPrice,
          side: pos.side,
          size: pos.size,
          pnl,
          fees: totalFees,
          netPnl,
          roe
        });

        equity += netPnl;

        // Reset grid level
        const level = gridLevels.find(l => l.level === pos.level);
        if (level) level.filled = false;

        positions.splice(p, 1);
      }
    }

    // Rebalance grid if price moved too far (every 500 candles)
    if (i % 500 === 0 && i > 0) {
      const priceMove = Math.abs(currentPrice - basePrice) / basePrice * 100;
      if (priceMove > gridSpacing * gridLevels_count * 0.4) {
        rebuildGrid(currentPrice);
      }
    }

    // Calculate unrealized PnL (depends on position side)
    let unrealizedPnL = 0;
    for (const pos of positions) {
      if (pos.side === 'LONG') {
        unrealizedPnL += (currentPrice - pos.entryPrice) * pos.size;
      } else {
        unrealizedPnL += (pos.entryPrice - currentPrice) * pos.size;
      }
    }

    const totalEquity = equity + unrealizedPnL;

    // Track drawdown
    if (totalEquity > maxEquity) maxEquity = totalEquity;
    const drawdown = maxEquity - totalEquity;
    if (drawdown > maxDrawdown) {
      maxDrawdown = drawdown;
      maxDrawdownPercent = (drawdown / maxEquity) * 100;
    }

    // Record equity and price curves (every 60 candles = 1 hour)
    if (i % 60 === 0) {
      equityCurve.push({ time: timestamp, equity: totalEquity });
      priceCurve.push({ time: timestamp, price: currentPrice });
    }

    // Update position size based on current equity (for auto strategy)
    if (config.autoStrategy && currentStrategy) {
      const params = STRATEGY_PRESETS[currentStrategy];
      positionSize = (equity * params.positionSizePercent) / 100;
    }
  }

  // Close remaining positions at last price (forced close = market order)
  const lastPrice = parseFloat(klines[klines.length - 1][4]);
  for (const pos of positions) {
    // PnL depends on position side
    const pnl = pos.side === 'LONG'
      ? (lastPrice - pos.entryPrice) * pos.size
      : (pos.entryPrice - lastPrice) * pos.size;
    // Entry was limit, exit is market (forced close)
    const entryFee = config.positionSize * config.leverage * MAKER_FEE;
    const exitFee = (pos.size * lastPrice) * TAKER_FEE;
    const totalFees = entryFee + exitFee;
    const netPnl = pnl - totalFees;
    const roe = (netPnl / config.positionSize) * 100;

    trades.push({
      entryTime: pos.entryTime,
      exitTime: klines[klines.length - 1][0],
      entryPrice: pos.entryPrice,
      exitPrice: lastPrice,
      side: pos.side,
      size: pos.size,
      pnl,
      fees: totalFees,
      netPnl,
      roe
    });

    equity += netPnl;
  }

  // Calculate summary stats
  const winningTrades = trades.filter(t => t.netPnl > 0);
  const losingTrades = trades.filter(t => t.netPnl <= 0);
  const totalPnL = trades.reduce((sum, t) => sum + t.pnl, 0);
  const totalFees = trades.reduce((sum, t) => sum + t.fees, 0);
  const netPnL = trades.reduce((sum, t) => sum + t.netPnl, 0);

  // Profit factor
  const grossProfit = winningTrades.reduce((sum, t) => sum + t.netPnl, 0);
  const grossLoss = Math.abs(losingTrades.reduce((sum, t) => sum + t.netPnl, 0));
  const profitFactor = grossLoss > 0 ? grossProfit / grossLoss : grossProfit > 0 ? Infinity : 0;

  // Consecutive wins/losses
  let maxConsecutiveWins = 0, maxConsecutiveLosses = 0;
  let currentWins = 0, currentLosses = 0;
  for (const t of trades) {
    if (t.netPnl > 0) {
      currentWins++;
      currentLosses = 0;
      if (currentWins > maxConsecutiveWins) maxConsecutiveWins = currentWins;
    } else {
      currentLosses++;
      currentWins = 0;
      if (currentLosses > maxConsecutiveLosses) maxConsecutiveLosses = currentLosses;
    }
  }

  // Monthly returns
  const monthlyReturns: { month: string; pnl: number; trades: number }[] = [];
  const monthlyMap = new Map<string, { pnl: number; trades: number }>();
  for (const t of trades) {
    const month = new Date(t.exitTime).toISOString().slice(0, 7);
    const existing = monthlyMap.get(month) || { pnl: 0, trades: 0 };
    existing.pnl += t.netPnl;
    existing.trades++;
    monthlyMap.set(month, existing);
  }
  for (const [month, data] of monthlyMap.entries()) {
    monthlyReturns.push({ month, ...data });
  }
  monthlyReturns.sort((a, b) => a.month.localeCompare(b.month));

  // Simple Sharpe approximation (daily returns)
  const dailyPnLs: number[] = [];
  let lastEquity = config.initialCapital;
  for (let i = 1; i < equityCurve.length; i++) {
    const dailyReturn = (equityCurve[i].equity - equityCurve[i-1].equity) / lastEquity;
    dailyPnLs.push(dailyReturn);
    lastEquity = equityCurve[i].equity;
  }
  const avgReturn = dailyPnLs.length > 0 ? dailyPnLs.reduce((a, b) => a + b, 0) / dailyPnLs.length : 0;
  const stdReturn = dailyPnLs.length > 1 ? Math.sqrt(dailyPnLs.reduce((sum, r) => sum + Math.pow(r - avgReturn, 2), 0) / dailyPnLs.length) : 1;
  const sharpeRatio = stdReturn > 0 ? (avgReturn / stdReturn) * Math.sqrt(365) : 0;

  return {
    symbol: config.symbol,
    period: { start: config.startDate, end: config.endDate },
    config,
    summary: {
      totalTrades: trades.length,
      winningTrades: winningTrades.length,
      losingTrades: losingTrades.length,
      winRate: trades.length > 0 ? (winningTrades.length / trades.length) * 100 : 0,
      totalPnL,
      totalFees,
      netPnL,
      maxDrawdown,
      maxDrawdownPercent,
      sharpeRatio,
      profitFactor,
      avgTradeProfit: trades.length > 0 ? netPnL / trades.length : 0,
      avgWinProfit: winningTrades.length > 0 ? grossProfit / winningTrades.length : 0,
      avgLossProfit: losingTrades.length > 0 ? -grossLoss / losingTrades.length : 0,
      maxConsecutiveWins,
      maxConsecutiveLosses
    },
    trades: trades.slice(-100),  // Last 100 trades
    equityCurve: equityCurve.filter((_, i) => i % 24 === 0),  // Daily equity
    priceCurve: priceCurve.filter((_, i) => i % 24 === 0),    // Daily price
    monthlyReturns,
    strategyChanges: config.autoStrategy ? strategyChanges : undefined,
    autoStrategyUsed: config.autoStrategy || false
  };
}

async function handleBacktest(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (req.method !== 'POST') {
    sendError(res, 'Method not allowed', 405);
    return;
  }

  try {
    const body = await parseBody(req);

    // Default config
    const config: BacktestConfig = {
      symbol: body.symbol || 'BTC',
      startDate: body.startDate || new Date(Date.now() - 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
      endDate: body.endDate || new Date().toISOString().split('T')[0],
      initialCapital: body.initialCapital || 10000,
      gridSpacing: body.gridSpacing || 0.5,
      gridLevels: body.gridLevels || 10,
      positionSize: body.positionSize || 100,
      leverage: body.leverage || 5,
      takeProfitPercent: body.takeProfitPercent || 0.5,
      autoStrategy: body.autoStrategy || false,
      bidirectional: body.bidirectional !== false // Default TRUE
    };

    console.log('[Backtest] Starting with config:', config);

    const result = await runGridBacktest(config);

    sendJson(res, {
      success: true,
      result
    });
  } catch (error: any) {
    console.error('[Backtest] Error:', error);
    sendError(res, `Backtest failed: ${error.message}`, 500);
  }
}

async function handleBacktestTick(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  if (req.method !== 'POST') {
    sendError(res, 'Method not allowed', 405);
    return;
  }

  try {
    const body = await parseBody(req);

    const config: BacktestConfig = {
      symbol: body.symbol || 'BTC',
      startDate: body.startDate || new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString().split('T')[0],
      endDate: body.endDate || new Date().toISOString().split('T')[0],
      initialCapital: body.initialCapital || 10000,
      gridSpacing: body.gridSpacing || 0.5,
      gridLevels: body.gridLevels || 5,
      positionSize: body.positionSize || 100,
      leverage: body.leverage || 5,
      takeProfitPercent: body.takeProfitPercent || 0.5,
      autoStrategy: false
    };

    console.log('[Backtest-Tick] Starting TICK-BY-TICK with config:', config);

    const result = await runGridBacktestTickByTick(config);

    sendJson(res, {
      success: true,
      mode: 'tick-by-tick',
      result
    });
  } catch (error: any) {
    console.error('[Backtest-Tick] Error:', error);
    sendError(res, `Tick backtest failed: ${error.message}`, 500);
  }
}

// ============= ROUTER =============

async function handleRequest(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  // CORS preflight
  if (req.method === 'OPTIONS') {
    res.writeHead(200, {
      'Access-Control-Allow-Origin': '*',
      'Access-Control-Allow-Methods': 'GET, POST, PUT, DELETE, OPTIONS',
      'Access-Control-Allow-Headers': 'Content-Type'
    });
    res.end();
    return;
  }

  const url = new URL(req.url || '/', 'http://localhost');
  const path = url.pathname;

  try {
    switch (path) {
      case '/api/grid/status':
      case '/grid/status':
        await handleStatus(req, res);
        break;

      case '/api/grid/start':
      case '/grid/start':
        if (req.method !== 'POST') { sendError(res, 'Method not allowed', 405); return; }
        await handleStart(req, res);
        break;

      case '/api/grid/stop':
      case '/grid/stop':
        if (req.method !== 'POST') { sendError(res, 'Method not allowed', 405); return; }
        await handleStop(req, res);
        break;

      case '/api/grid/pause':
      case '/grid/pause':
        if (req.method !== 'POST') { sendError(res, 'Method not allowed', 405); return; }
        await handlePause(req, res);
        break;

      case '/api/grid/resume':
      case '/grid/resume':
        if (req.method !== 'POST') { sendError(res, 'Method not allowed', 405); return; }
        await handleResume(req, res);
        break;

      case '/api/grid/rebalance':
      case '/grid/rebalance':
        if (req.method !== 'POST') { sendError(res, 'Method not allowed', 405); return; }
        await handleRebalance(req, res);
        break;

      case '/api/grid/close-all':
      case '/grid/close-all':
        if (req.method !== 'POST') { sendError(res, 'Method not allowed', 405); return; }
        await handleCloseAll(req, res);
        break;

      case '/api/grid/positions':
      case '/grid/positions':
        await handlePositions(req, res);
        break;

      case '/api/grid/trades':
      case '/grid/trades':
        await handleTrades(req, res);
        break;

      case '/api/grid/orders':
      case '/grid/orders':
        await handleOpenOrders(req, res);
        break;

      case '/api/grid/levels':
      case '/grid/levels':
        await handleGridLevels(req, res);
        break;

      case '/api/grid/config':
      case '/grid/config':
        await handleConfig(req, res);
        break;

      case '/api/grid/presets':
      case '/grid/presets':
        await handlePresets(req, res);
        break;

      case '/api/grid/health':
      case '/grid/health':
      case '/health':
        await handleHealth(req, res);
        break;

      case '/api/grid/backtest':
      case '/grid/backtest':
        await handleBacktest(req, res);
        break;

      case '/api/grid/backtest-tick':
      case '/grid/backtest-tick':
        await handleBacktestTick(req, res);
        break;

      case '/api/grid/test-connection':
      case '/grid/test-connection':
        await handleTestConnection(req, res);
        break;

      case '/api/grid/saved-config':
      case '/grid/saved-config':
        await handleSavedConfig(req, res);
        break;

      case '/api/grid/kraken-positions':
      case '/grid/kraken-positions':
        await handleKrakenPositions(req, res);
        break;

      case '/api/grid/place-tpsl':
      case '/grid/place-tpsl':
        await handlePlaceTPSL(req, res);
        break;

      case '/api/grid/cancel-order':
      case '/grid/cancel-order':
        await handleCancelOrder(req, res);
        break;

      case '/api/grid/close-positions':
      case '/grid/close-positions':
        await handleClosePositions(req, res);
        break;

      default:
        sendError(res, 'Not found', 404);
    }
  } catch (error: any) {
    console.error('[API] Error:', error);
    sendError(res, error.message, 500);
  }
}

// ============= CANCEL SPECIFIC ORDER ON KRAKEN =============

async function cancelKrakenOrder(mode: string, apiKey: string, apiSecret: string, orderId: string): Promise<any> {
  const crypto = await import('crypto');
  const baseUrl = mode === 'live'
    ? 'https://futures.kraken.com/derivatives/api/v3'
    : 'https://demo-futures.kraken.com/derivatives/api/v3';

  const endpoint = '/cancelorder';
  const fullEndpoint = '/api/v3' + endpoint;
  const nonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
  const postData = `order_id=${orderId}`;

  const message = postData + nonce + fullEndpoint;
  const hash = crypto.createHash('sha256').update(message).digest();
  const signature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64'))
    .update(hash)
    .digest('base64');

  const response = await fetch(`${baseUrl}${endpoint}`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
      'APIKey': apiKey,
      'Nonce': nonce,
      'Authent': signature,
    },
    body: postData
  });

  return response.json();
}

async function handleClosePositions(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  const url = new URL(req.url || '/', `http://${req.headers.host}`);
  const mode = url.searchParams.get('mode') || 'live';

  const apiKey = mode === 'live'
    ? (process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY)
    : process.env.KRAKEN_DEMO_API_KEY;
  const apiSecret = mode === 'live'
    ? (process.env.KRAKEN_FUTURES_API_SECRET || process.env.KRAKEN_API_SECRET)
    : process.env.KRAKEN_DEMO_API_SECRET;

  if (!apiKey || !apiSecret) {
    sendError(res, `API keys not configured for ${mode} mode`, 400);
    return;
  }

  const crypto = await import('crypto');
  const baseUrl = mode === 'live'
    ? 'https://futures.kraken.com/derivatives/api/v3'
    : 'https://demo-futures.kraken.com/derivatives/api/v3';

  // First cancel all orders
  const cancelEndpoint = '/cancelallorders';
  const cancelFullEndpoint = '/api/v3' + cancelEndpoint;
  const cancelNonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
  const cancelMessage = '' + cancelNonce + cancelFullEndpoint;
  const cancelHash = crypto.createHash('sha256').update(cancelMessage).digest();
  const cancelSignature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64')).update(cancelHash).digest('base64');

  await fetch(`${baseUrl}${cancelEndpoint}`, {
    method: 'POST',
    headers: { 'APIKey': apiKey, 'Nonce': cancelNonce, 'Authent': cancelSignature }
  });

  // Get open positions
  const posEndpoint = '/openpositions';
  const posFullEndpoint = '/api/v3' + posEndpoint;
  const posNonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
  const posMessage = '' + posNonce + posFullEndpoint;
  const posHash = crypto.createHash('sha256').update(posMessage).digest();
  const posSignature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64')).update(posHash).digest('base64');

  const posResponse = await fetch(`${baseUrl}${posEndpoint}`, {
    method: 'GET',
    headers: { 'APIKey': apiKey, 'Nonce': posNonce, 'Authent': posSignature }
  });
  const posData = await posResponse.json() as any;
  const positions = posData.openPositions || [];

  const results: any[] = [];

  // Close each position with market order
  for (const pos of positions) {
    const closeSide = pos.side === 'long' ? 'sell' : 'buy';

    try {
      const result = await placeKrakenOrder(mode, apiKey, apiSecret, {
        symbol: pos.symbol,
        side: closeSide,
        size: pos.size,
        orderType: 'mkt',
        reduceOnly: true
      });
      results.push({
        symbol: pos.symbol,
        side: pos.side,
        size: pos.size,
        closed: result.result === 'success',
        orderId: result.sendStatus?.order_id,
        error: result.error
      });
    } catch (e: any) {
      results.push({ symbol: pos.symbol, error: e.message });
    }
  }

  sendJson(res, {
    success: true,
    ordersCancelled: true,
    positionsClosed: results.length,
    results
  });
}

async function handleCancelOrder(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  const url = new URL(req.url || '/', `http://${req.headers.host}`);
  const orderId = url.searchParams.get('orderId');

  if (!orderId) {
    sendError(res, 'orderId parameter required', 400);
    return;
  }

  const apiKey = process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY;
  const apiSecret = process.env.KRAKEN_FUTURES_API_SECRET || process.env.KRAKEN_API_SECRET;

  if (!apiKey || !apiSecret) {
    sendError(res, 'API keys not configured', 400);
    return;
  }

  try {
    const result = await cancelKrakenOrder('live', apiKey, apiSecret, orderId);
    sendJson(res, { success: result.result === 'success', result });
  } catch (e: any) {
    sendError(res, e.message, 500);
  }
}

// ============= PLACE ORDER ON KRAKEN =============

interface PlaceOrderParams {
  symbol: string;
  side: 'buy' | 'sell';
  size: number;
  orderType: 'lmt' | 'stp' | 'mkt';  // limit, stop, market
  limitPrice?: number;
  stopPrice?: number;
  reduceOnly?: boolean;
}

async function placeKrakenOrder(mode: string, apiKey: string, apiSecret: string, params: PlaceOrderParams): Promise<any> {
  const crypto = await import('crypto');
  const baseUrl = mode === 'live'
    ? 'https://futures.kraken.com/derivatives/api/v3'
    : 'https://demo-futures.kraken.com/derivatives/api/v3';

  const endpoint = '/sendorder';
  const fullEndpoint = '/api/v3' + endpoint;
  const nonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();

  // Build post data
  const postParams: Record<string, string> = {
    orderType: params.orderType,
    symbol: params.symbol,
    side: params.side,
    size: params.size.toString(),
  };

  if (params.limitPrice) postParams.limitPrice = params.limitPrice.toString();
  if (params.stopPrice) postParams.stopPrice = params.stopPrice.toString();
  if (params.reduceOnly) postParams.reduceOnly = 'true';

  const postData = new URLSearchParams(postParams).toString();

  const message = postData + nonce + fullEndpoint;
  const hash = crypto.createHash('sha256').update(message).digest();
  const signature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64'))
    .update(hash)
    .digest('base64');

  const response = await fetch(`${baseUrl}${endpoint}`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
      'APIKey': apiKey,
      'Nonce': nonce,
      'Authent': signature,
    },
    body: postData
  });

  return response.json();
}

async function handlePlaceTPSL(req: http.IncomingMessage, res: http.ServerResponse): Promise<void> {
  const mode = 'live';
  const apiKey = process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY;
  const apiSecret = process.env.KRAKEN_FUTURES_API_SECRET || process.env.KRAKEN_API_SECRET;

  if (!apiKey || !apiSecret) {
    sendError(res, 'API keys not configured', 400);
    return;
  }

  // Parse body for custom TP/SL levels
  let body: any = {};
  if (req.method === 'POST') {
    const chunks: Buffer[] = [];
    for await (const chunk of req) chunks.push(chunk as Buffer);
    const rawBody = Buffer.concat(chunks).toString();
    if (rawBody) body = JSON.parse(rawBody);
  }

  // Default TP/SL percentages
  const tpPercent = body.tpPercent || 0.5;  // +0.5% for TP
  const slPercent = body.slPercent || 2.0;  // -2% for SL

  const results: any[] = [];

  // Get current positions
  const crypto = await import('crypto');
  const baseUrl = 'https://futures.kraken.com/derivatives/api/v3';

  // Query positions
  const posEndpoint = '/openpositions';
  const posFullEndpoint = '/api/v3' + posEndpoint;
  const posNonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
  const posMessage = '' + posNonce + posFullEndpoint;
  const posHash = crypto.createHash('sha256').update(posMessage).digest();
  const posSignature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64')).update(posHash).digest('base64');

  const posResponse = await fetch(`${baseUrl}${posEndpoint}`, {
    method: 'GET',
    headers: { 'APIKey': apiKey, 'Nonce': posNonce, 'Authent': posSignature }
  });
  const posData = await posResponse.json() as any;
  const positions = posData.openPositions || [];

  // Query existing orders to check for existing TP/SL
  const ordEndpoint = '/openorders';
  const ordFullEndpoint = '/api/v3' + ordEndpoint;
  const ordNonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
  const ordMessage = '' + ordNonce + ordFullEndpoint;
  const ordHash = crypto.createHash('sha256').update(ordMessage).digest();
  const ordSignature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64')).update(ordHash).digest('base64');

  const ordResponse = await fetch(`${baseUrl}${ordEndpoint}`, {
    method: 'GET',
    headers: { 'APIKey': apiKey, 'Nonce': ordNonce, 'Authent': ordSignature }
  });
  const ordData = await ordResponse.json() as any;
  const existingOrders = ordData.openOrders || [];

  for (const pos of positions) {
    const symbol = pos.symbol;
    const side = pos.side;
    const size = pos.size;
    const entryPrice = pos.price;

    // Find existing reduceOnly orders for this symbol
    const existingReduceOnly = existingOrders.filter((o: any) =>
      o.symbol === symbol && o.reduceOnly === true
    );
    const existingTPSize = existingReduceOnly
      .filter((o: any) => o.orderType === 'lmt')
      .reduce((sum: number, o: any) => sum + (o.unfilledSize || 0), 0);
    const existingSLSize = existingReduceOnly
      .filter((o: any) => o.orderType === 'stp' || o.orderType === 'stop')
      .reduce((sum: number, o: any) => sum + (o.unfilledSize || 0), 0);

    // Calculate TP/SL prices based on position side
    let tpPrice: number, slPrice: number, closeSide: 'buy' | 'sell';

    if (side === 'long') {
      tpPrice = Math.round(entryPrice * (1 + tpPercent / 100) * 100) / 100;
      slPrice = Math.round(entryPrice * (1 - slPercent / 100) * 100) / 100;
      closeSide = 'sell';
    } else {
      tpPrice = Math.round(entryPrice * (1 - tpPercent / 100) * 100) / 100;
      slPrice = Math.round(entryPrice * (1 + slPercent / 100) * 100) / 100;
      closeSide = 'buy';
    }

    // Round prices for BTC
    if (symbol.includes('XBT')) {
      tpPrice = Math.round(tpPrice);
      slPrice = Math.round(slPrice);
    }

    // Place TP if not fully covered
    const tpNeeded = size - existingTPSize;
    if (tpNeeded > 0.0001) {
      try {
        const tpResult = await placeKrakenOrder(mode, apiKey, apiSecret, {
          symbol,
          side: closeSide,
          size: Math.round(tpNeeded * 10000) / 10000,
          orderType: 'lmt',
          limitPrice: tpPrice,
          reduceOnly: true
        });
        results.push({
          type: 'TP',
          symbol,
          size: tpNeeded,
          price: tpPrice,
          result: tpResult.result,
          orderId: tpResult.sendStatus?.order_id,
          error: tpResult.error
        });
      } catch (e: any) {
        results.push({ type: 'TP', symbol, error: e.message });
      }
    } else {
      results.push({ type: 'TP', symbol, status: 'already_covered', existingSize: existingTPSize });
    }

    // Place SL if not fully covered
    const slNeeded = size - existingSLSize;
    if (slNeeded > 0.0001) {
      try {
        const slResult = await placeKrakenOrder(mode, apiKey, apiSecret, {
          symbol,
          side: closeSide,
          size: Math.round(slNeeded * 10000) / 10000,
          orderType: 'stp',
          stopPrice: slPrice,
          reduceOnly: true
        });
        results.push({
          type: 'SL',
          symbol,
          size: slNeeded,
          price: slPrice,
          result: slResult.result,
          orderId: slResult.sendStatus?.order_id,
          error: slResult.error
        });
      } catch (e: any) {
        results.push({ type: 'SL', symbol, error: e.message });
      }
    } else {
      results.push({ type: 'SL', symbol, status: 'already_covered', existingSize: existingSLSize });
    }
  }

  sendJson(res, {
    success: true,
    tpPercent,
    slPercent,
    positionsProcessed: positions.length,
    results
  });
}

// ============= CANCEL ALL KRAKEN ORDERS =============

async function cancelAllKrakenOrdersWithCount(mode: string, apiKey: string, apiSecret: string): Promise<number> {
  const crypto = await import('crypto');
  const baseUrl = mode === 'live'
    ? 'https://futures.kraken.com/derivatives/api/v3'
    : 'https://demo-futures.kraken.com/derivatives/api/v3';

  const endpoint = '/cancelallorders';
  const fullEndpoint = '/api/v3' + endpoint;
  const nonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
  const postData = '';

  const message = postData + nonce + fullEndpoint;
  const hash = crypto.createHash('sha256').update(message).digest();
  const signature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64'))
    .update(hash)
    .digest('base64');

  try {
    const response = await fetch(`${baseUrl}${endpoint}`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'APIKey': apiKey,
        'Nonce': nonce,
        'Authent': signature,
      }
    });

    const data = await response.json() as any;
    if (data.result === 'success') {
      return data.cancelStatus?.cancelledOrders?.length || 0;
    }
  } catch (error: any) {
    console.error(`[GridAPI] Failed to cancel orders: ${error.message}`);
  }
  return 0;
}

// ============= AUTO-RESTART FROM SAVED CONFIG =============

async function autoRestartFromSavedConfig(): Promise<void> {
  const savedConfig = loadSavedConfig();

  if (!savedConfig || !savedConfig.autoRestart) {
    console.log('[GridAPI] No saved config or auto-restart disabled');
    return;
  }

  console.log(`[GridAPI] Found saved config from ${savedConfig.savedAt}`);
  console.log(`[GridAPI] Auto-restarting ${savedConfig.mode.toUpperCase()} mode with ${savedConfig.symbols.length} symbols...`);

  try {
    // Get API keys based on mode
    const mode = savedConfig.mode;
    let apiKey: string | undefined, apiSecret: string | undefined;

    if (mode === 'live') {
      apiKey = process.env.KRAKEN_FUTURES_API_KEY || process.env.KRAKEN_API_KEY;
      apiSecret = process.env.KRAKEN_FUTURES_API_SECRET || process.env.KRAKEN_API_SECRET;
    } else {
      apiKey = process.env.KRAKEN_DEMO_API_KEY;
      apiSecret = process.env.KRAKEN_DEMO_API_SECRET;
    }

    if (!apiKey || !apiSecret) {
      console.error(`[GridAPI] Auto-restart failed: API keys not configured for ${mode} mode`);
      return;
    }

    // Cancel existing orders before placing new ones
    console.log(`[GridAPI] Cancelling existing orders before auto-restart...`);
    const cancelledCount = await cancelAllKrakenOrdersWithCount(mode, apiKey, apiSecret);
    console.log(`[GridAPI] Cancelled ${cancelledCount} existing orders`);

    const config: Partial<GridEngineConfig> = {
      mode: savedConfig.mode as any,
      symbols: savedConfig.symbols,
      initialCapital: savedConfig.initialCapital,
      gridSpacing: savedConfig.gridSpacing,
      gridLevels: savedConfig.gridLevels,
      positionPerGrid: savedConfig.positionPerGrid,
      leverage: savedConfig.leverage,
      maxDrawdownPercent: savedConfig.maxDrawdownPercent,
      useLimitOrders: savedConfig.useLimitOrders,
      autoStrategy: savedConfig.autoStrategy,
      bidirectional: savedConfig.bidirectional ?? true,
      autoBidirectional: savedConfig.autoBidirectional ?? false,
      apiKey,
      apiSecret
    };

    gridEngine = new GridTradingEngine(config);

    // Setup event listeners
    gridEngine.on('started', (data) => console.log('[API] Grid engine auto-started:', data));
    gridEngine.on('stopped', () => console.log('[API] Grid engine stopped'));
    gridEngine.on('position_opened', (pos) => console.log(`[API] Position opened: ${pos.symbol} @ ${pos.entryPrice}`));
    gridEngine.on('position_closed', (trade) => console.log(`[API] Position closed: ${trade.symbol} PnL=${trade.netPnl.toFixed(2)}`));
    gridEngine.on('circuit_breaker', (data) => console.log(`[API] Circuit breaker: ${data.symbol} DD=${data.drawdown.toFixed(1)}%`));
    gridEngine.on('strategy_changed', (data) => console.log(`[API] Strategy changed: ${data.from || 'none'} -> ${data.to} (${data.reason})`));
    gridEngine.on('bidirectional_change', (data) => console.log(`[API] Bidirectional change: ${data.symbol} - ${data.reason}`));
    gridEngine.on('grid_rebalanced', (data) => console.log(`[API] Dynamic Grid Rebalanced: ${data.symbol} ${data.oldBase.toFixed(2)} -> ${data.newBase.toFixed(2)} (${data.deviation.toFixed(2)}%)`));

    await gridEngine.start();

    console.log(`[GridAPI] ✅ Auto-restart successful! ${savedConfig.mode.toUpperCase()} mode running`);
    console.log(`[GridAPI]    Symbols: ${savedConfig.symbols.join(', ')}`);
    console.log(`[GridAPI]    Leverage: ${savedConfig.leverage}x`);
    console.log(`[GridAPI]    Grid: ${savedConfig.gridLevels} levels @ ${savedConfig.gridSpacing}% spacing`);
  } catch (error: any) {
    console.error(`[GridAPI] Auto-restart failed: ${error.message}`);
    // Don't clear config on auto-restart failure - user might want to try again manually
  }
}

// ============= SERVER STARTUP =============

export function startGridTradingApi(port: number = 3032): http.Server {
  const server = http.createServer(handleRequest);

  server.listen(port, '0.0.0.0', () => {
    console.log(`[GridAPI] Server listening on port ${port}`);
    console.log(`[GridAPI] Endpoints:`);
    console.log(`  GET  /grid/status     - Get current status`);
    console.log(`  POST /grid/start      - Start grid trading`);
    console.log(`  POST /grid/stop       - Stop grid trading`);
    console.log(`  POST /grid/pause      - Pause trading`);
    console.log(`  POST /grid/resume     - Resume trading`);
    console.log(`  POST /grid/rebalance  - Rebalance grids`);
    console.log(`  POST /grid/close-all  - Close all positions`);
    console.log(`  GET  /grid/positions  - Get open positions`);
    console.log(`  GET  /grid/trades     - Get trade history`);
    console.log(`  GET  /grid/levels     - Get grid levels`);
    console.log(`  GET  /grid/config     - Get/update config`);
    console.log(`  GET  /grid/presets    - Get available presets`);

    // Auto-restart from saved config after a short delay
    setTimeout(() => {
      autoRestartFromSavedConfig().catch(e => console.error('[GridAPI] Auto-restart error:', e));
    }, 2000);
  });

  return server;
}

// Run server if called directly
if (import.meta.url === `file://${process.argv[1]}`) {
  const port = parseInt(process.env.GRID_API_PORT || '3037');
  startGridTradingApi(port);
}

export default startGridTradingApi;
