/**
 * GRID TRADING ENGINE - Kraken Integration
 *
 * Real-time grid trading with Kraken Futures API
 *
 * === THREE TRADING MODES ===
 *
 * TEST MODE:
 * - Prices: WebSocket from BINANCE (real-time tick-by-tick, same as backtest)
 * - Balance/Equity: Simulated locally (initialCapital)
 * - Orders: Simulated locally - Kraken-identical order matching engine
 * - Positions: Tracked locally (simulates Kraken exactly)
 * - Fees: Simulated (maker 0.02%, taker 0.05%)
 * - Risk: None (pure simulation with real market data)
 *
 * DEMO MODE:
 * - Prices: WebSocket from demo-futures.kraken.com
 * - Balance/Equity: From Kraken Demo account (API)
 * - Orders: Real orders on Kraken Demo exchange
 * - Positions: From Kraken API (real tracking)
 * - Risk: None (demo money)
 *
 * LIVE MODE:
 * - Prices: WebSocket from futures.kraken.com
 * - Balance/Equity: From Kraken Live account (API)
 * - Orders: Real orders with real money
 * - Positions: From Kraken API (real tracking)
 * - Risk: HIGH (real money at stake)
 *
 * Kraken Futures Fees:
 * - Maker: 0.02% (limit orders)
 * - Taker: 0.05% (market orders)
 * - Funding: Variable (checked via API)
 */

import { EventEmitter } from 'events';
import WebSocket from 'ws';
import crypto from 'crypto';
import axios from 'axios';
import { getSymbol, getKrakenSymbol, roundSize, roundPrice, validateLeverage, KRAKEN_SYMBOLS } from './core/symbols.js';
import type { TradingMode } from './core/types.js';
import { analyzeMarketConditions, getStrategyParams, STRATEGY_PRESETS, type MarketAnalysis, type StrategyPreset } from './market-analyzer.js';
import { BinanceHistoricalData } from './backtester.js';

// ============= TYPES =============

export interface GridEngineConfig {
  mode: TradingMode;                    // 'test' | 'demo' | 'live'
  symbols: string[];                    // ['BTC', 'ETH']
  initialCapital: number;               // Starting capital in USD
  capitalPerSymbol?: number;            // Override: capital per symbol
  gridSpacing: number;                  // % between grid levels (0.4-0.8)
  gridLevels: number;                   // Levels above/below (6-12)
  positionPerGrid: number;              // $ per grid level
  leverage: number;                     // 3-8x recommended
  maxOpenPositions: number;             // Max concurrent positions per symbol
  maxDrawdownPercent: number;           // Circuit breaker (30%)
  trendFilterEnabled: boolean;          // Pause during strong trends
  trendFilterThreshold: number;         // % move to pause (3%)
  useLimitOrders: boolean;              // true = maker fees (0.02%)
  autoStrategy?: boolean;               // Auto-adjust strategy based on market
  bidirectional?: boolean;              // Grid LONG + SHORT (above and below price)
  immediateEntry?: boolean;             // Open position immediately with market order at start
  autoBidirectional?: boolean;          // Auto enable/disable SHORT based on market trend
  useSimulatedCapital?: boolean;        // Use initialCapital instead of real Kraken balance
  // Dynamic Grid (trailing grid that follows price)
  dynamicGrid?: boolean;                // Enable dynamic grid that recenters when price moves
  gridRebalanceThreshold?: number;      // % deviation from base to trigger rebalance (default: 2%)
  // HYPER-DYNAMIC GRID (ATR-based adaptive system)
  hyperDynamic?: boolean;               // Enable hyper-dynamic mode with ATR-based spacing
  atrMultiplier?: number;               // Grid spacing = ATR% × multiplier (default: 0.5)
  minGridSpacing?: number;              // Minimum grid spacing % (default: 0.15)
  maxGridSpacing?: number;              // Maximum grid spacing % (default: 1.5)
  volatilityThreshold?: number;         // Min ATR% to trade (default: 0.3, pause if lower)
  tickRecalibration?: boolean;          // Recalibrate on every price tick (default: true if hyperDynamic)
  // API Keys - Demo (for demo mode)
  apiKey?: string;                      // Demo API Key
  apiSecret?: string;                   // Demo API Secret
  // API Keys - Live (for live mode only)
  liveApiKey?: string;                  // Live API Key (REQUIRED for live mode)
  liveApiSecret?: string;               // Live API Secret (REQUIRED for live mode)
  // TEST MODE (Backtest) parameters
  testStartDate?: string;               // Backtest start date (ISO format)
  testEndDate?: string;                 // Backtest end date (ISO format)
  testCandleInterval?: string;          // Candle interval: '1m', '5m', '15m', '1h', etc.
  testSpeed?: number;                   // Simulation speed multiplier (1 = realtime, 100 = 100x faster)
}

export interface GridState {
  symbol: string;
  basePrice: number;
  currentPrice: number;
  gridLevels: GridLevel[];
  positions: GridPosition[];
  trades: GridTrade[];
  equity: number;
  realizedPnL: number;
  unrealizedPnL: number;
  maxDrawdown: number;
  totalFees: number;
  isPaused: boolean;
  pauseReason: string | null;
}

export interface GridLevel {
  price: number;
  level: number;
  isBuyLevel: boolean;
  side: 'LONG' | 'SHORT';  // LONG = buy low sell high, SHORT = sell high buy low
  filled: boolean;
  orderId?: string;       // Kraken order ID when order placed
  orderStatus?: 'pending' | 'filled' | 'cancelled';
}

export interface GridPosition {
  id: string;
  symbol: string;
  side: 'LONG' | 'SHORT';
  entryPrice: number;
  size: number;
  margin: number;
  gridLevel: number;
  tpPrice: number;
  tpOrderId?: string;
  entryFee: number;
  entryTime: Date;
  krakenOrderId?: string;
}

export interface GridTrade {
  id: string;
  symbol: string;
  side: 'LONG' | 'SHORT';
  entryPrice: number;
  exitPrice: number;
  pnl: number;
  fees: number;
  netPnl: number;
  entryTime: Date;
  exitTime: Date;
  result: 'TP' | 'SL' | 'CLOSE';
}

interface KrakenTicker {
  symbol: string;
  bid: number;
  ask: number;
  last: number;
  volume: number;
  change24h: number;
}

// Per-symbol bidirectional state for auto-bidirectional mode
interface SymbolBidirectionalState {
  shortsEnabled: boolean;
  longsEnabled: boolean;
  currentTrend: 'bullish' | 'bearish' | 'neutral';
  trendStrength: number;  // 0-100
  lastTrendCheck: number;
  reason: string;
}

// ============= GRID TRADING ENGINE =============

export class GridTradingEngine extends EventEmitter {
  private config: GridEngineConfig;
  private ws: WebSocket | null = null;
  private states: Map<string, GridState> = new Map();
  private tickers: Map<string, KrakenTicker> = new Map();
  private priceHistory: Map<string, number[]> = new Map();
  private isRunning: boolean = false;
  private startTime: number = 0;
  private tradeIdCounter: number = 0;
  private reconnectAttempts: number = 0;
  private heartbeatInterval: NodeJS.Timeout | null = null;
  private initialCapital: number = 0;
  private testBalance: number = 0;  // For paper trading
  private currentStrategy: StrategyPreset | null = null;
  private strategyAnalysisInterval: NodeJS.Timeout | null = null;
  private lastStrategyChange: number = 0;
  private krakenEquity: number = 0;
  private krakenAvailableMargin: number = 0;
  private balanceUpdateInterval: NodeJS.Timeout | null = null;
  private orderSyncInterval: NodeJS.Timeout | null = null;
  private pendingOrders: Map<string, { orderId: string; symbol: string; side: 'buy' | 'sell'; size: number; price: number; gridLevel: number; placedAt: Date; positionId?: string }> = new Map();
  private krakenPositions: Map<string, { symbol: string; side: string; size: number; price: number; pnl: number }> = new Map();

  // Auto-bidirectional state per symbol
  private symbolBidirectionalState: Map<string, SymbolBidirectionalState> = new Map();
  private autoBidirectionalInterval: NodeJS.Timeout | null = null;

  // Dynamic grid tracking
  private lastGridRebalance: Map<string, number> = new Map();
  private readonly MIN_REBALANCE_INTERVAL = 60000; // Minimum 60 seconds between rebalances

  // HYPER-DYNAMIC: OHLC tracking for ATR calculation
  private ohlcData: Map<string, { open: number; high: number; low: number; close: number; time: number }[]> = new Map();
  private currentCandle: Map<string, { open: number; high: number; low: number; close: number; startTime: number }> = new Map();
  private readonly CANDLE_INTERVAL_MS = 60000; // 1-minute candles for ATR
  private currentATR: Map<string, number> = new Map();
  private currentATRPercent: Map<string, number> = new Map();
  private lastHyperRecalibration: Map<string, number> = new Map();
  private readonly MIN_HYPER_RECALIBRATION_INTERVAL = 5000; // 5 seconds between recalibrations

  // Kraken API
  private readonly API_URL_LIVE = 'https://futures.kraken.com/derivatives/api/v3';
  private readonly API_URL_DEMO = 'https://demo-futures.kraken.com/derivatives/api/v3';
  private readonly WS_URL_LIVE = 'wss://futures.kraken.com/ws/v1';
  private readonly WS_URL_DEMO = 'wss://demo-futures.kraken.com/ws/v1';
  // Binance WebSocket for TEST mode (real-time prices)
  private readonly WS_URL_BINANCE = 'wss://fstream.binance.com/ws';

  constructor(config: Partial<GridEngineConfig>) {
    super();

    this.config = {
      mode: config.mode || 'demo',  // Default to demo (real Kraken orders)
      symbols: config.symbols || ['BTC', 'ETH'],
      initialCapital: config.initialCapital || 3000,
      gridSpacing: config.gridSpacing || 0.5,
      gridLevels: config.gridLevels || 10,
      positionPerGrid: config.positionPerGrid || 100,
      leverage: config.leverage || 5,
      maxOpenPositions: config.maxOpenPositions || 10,
      maxDrawdownPercent: config.maxDrawdownPercent || 30,
      trendFilterEnabled: config.trendFilterEnabled ?? true,
      trendFilterThreshold: config.trendFilterThreshold || 3,
      useLimitOrders: config.useLimitOrders ?? true,
      autoStrategy: config.autoStrategy ?? false,
      dynamicGrid: config.dynamicGrid ?? false,
      gridRebalanceThreshold: config.gridRebalanceThreshold || 2,  // 2% default
      // HYPER-DYNAMIC defaults
      hyperDynamic: config.hyperDynamic ?? false,
      atrMultiplier: config.atrMultiplier || 0.5,          // Grid = ATR% × 0.5
      minGridSpacing: config.minGridSpacing || 0.15,       // Min 0.15%
      maxGridSpacing: config.maxGridSpacing || 1.5,        // Max 1.5%
      volatilityThreshold: config.volatilityThreshold || 0.3, // Pause below 0.3% ATR
      tickRecalibration: config.tickRecalibration ?? true,
      // Demo API Keys (used for demo mode)
      apiKey: config.apiKey || process.env.KRAKEN_DEMO_API_KEY,
      apiSecret: config.apiSecret || process.env.KRAKEN_DEMO_API_SECRET,
      // Live API Keys (used for live mode only)
      liveApiKey: config.liveApiKey || process.env.KRAKEN_LIVE_API_KEY || process.env.KRAKEN_FUTURES_API_KEY,
      liveApiSecret: config.liveApiSecret || process.env.KRAKEN_LIVE_API_SECRET || process.env.KRAKEN_FUTURES_API_SECRET,
      ...config
    };

    // Validate API keys for non-test modes
    if (this.config.mode === 'live') {
      if (!this.config.liveApiKey || !this.config.liveApiSecret) {
        throw new Error('LIVE mode requires liveApiKey and liveApiSecret');
      }
    } else if (this.config.mode === 'demo') {
      if (!this.config.apiKey || !this.config.apiSecret) {
        throw new Error('DEMO mode requires apiKey and apiSecret');
      }
    }
    // Paper mode doesn't need API keys

    this.initialCapital = this.config.initialCapital;
    this.testBalance = this.initialCapital;

    // Capital per symbol will be calculated after auto-scaling
    // Initialize state for each symbol
    for (const symbol of this.config.symbols) {
      this.states.set(symbol, this.createInitialState(symbol));
      this.priceHistory.set(symbol, []);
      // Initialize OHLC tracking for hyper-dynamic mode
      this.ohlcData.set(symbol, []);
      this.currentATR.set(symbol, 0);
      this.currentATRPercent.set(symbol, 0);
    }
  }

  /**
   * AUTO-SCALING: Calculate optimal configuration based on available equity
   * Prioritizes: more symbols > more grid levels > larger position size
   */
  private autoScaleConfig(availableEquity: number): void {
    console.log(`[AutoScale] ====================================`);
    console.log(`[AutoScale] Calculating optimal config for $${availableEquity.toFixed(2)} equity`);

    const leverage = this.config.leverage;
    const requestedSymbols = this.config.symbols.length;
    const requestedLevels = this.config.gridLevels;
    const requestedPositionSize = this.config.positionPerGrid;

    // Minimum requirements
    const MIN_POSITION_SIZE = 5;   // $5 minimum per order
    const MIN_GRID_LEVELS = 2;      // At least 2 levels
    const MIN_SYMBOLS = 1;          // At least 1 symbol
    const MARGIN_BUFFER = 0.85;     // Use 85% of available margin (15% safety buffer)

    const usableMargin = availableEquity * MARGIN_BUFFER;

    // Calculate margin needed per order
    // Kraken uses: Initial Margin = NotionalValue / MaxLeverage
    // For a $25 position at 10x leverage, margin = $25 * 10 / MaxLev (usually 50) ≈ $5
    // But they also reserve margin for pending orders, so use conservative estimate:
    // Margin per order ≈ positionSize (without leverage multiplier for safety)
    // This is conservative but ensures we don't run out of margin

    let optimalSymbols = requestedSymbols;
    let optimalLevels = requestedLevels;
    let optimalPositionSize = requestedPositionSize;

    // CONSERVATIVE margin calculation:
    // Each order requires positionSize as margin reserve (Kraken holds full notional for pending orders)
    const calcMargin = (syms: number, levels: number, posSize: number) => syms * levels * posSize;

    // Try to fit all requested config first
    let marginNeeded = calcMargin(optimalSymbols, optimalLevels, optimalPositionSize);

    console.log(`[AutoScale] Requested: ${optimalSymbols} symbols × ${optimalLevels} levels × $${optimalPositionSize}/order`);
    console.log(`[AutoScale] Margin needed: $${marginNeeded.toFixed(2)}, Available: $${usableMargin.toFixed(2)}`);

    if (marginNeeded <= usableMargin) {
      console.log(`[AutoScale] ✅ Requested config fits within margin`);
    } else {
      console.log(`[AutoScale] ⚠️ Adjusting config to fit available margin...`);

      // Strategy 1: Reduce position size first (down to minimum)
      while (marginNeeded > usableMargin && optimalPositionSize > MIN_POSITION_SIZE) {
        optimalPositionSize = Math.max(MIN_POSITION_SIZE, optimalPositionSize - 5);
        marginNeeded = calcMargin(optimalSymbols, optimalLevels, optimalPositionSize);
      }

      // Strategy 2: Reduce grid levels (down to minimum)
      while (marginNeeded > usableMargin && optimalLevels > MIN_GRID_LEVELS) {
        optimalLevels--;
        marginNeeded = calcMargin(optimalSymbols, optimalLevels, optimalPositionSize);
      }

      // Strategy 3: Reduce symbols (down to minimum)
      while (marginNeeded > usableMargin && optimalSymbols > MIN_SYMBOLS) {
        optimalSymbols--;
        marginNeeded = calcMargin(optimalSymbols, optimalLevels, optimalPositionSize);
      }

      // Final check - if still not enough, reduce position size to absolute minimum
      if (marginNeeded > usableMargin) {
        optimalPositionSize = Math.floor(usableMargin / (optimalSymbols * optimalLevels));
        optimalPositionSize = Math.max(MIN_POSITION_SIZE, optimalPositionSize);
        marginNeeded = calcMargin(optimalSymbols, optimalLevels, optimalPositionSize);
      }
    }

    // Apply optimized config
    if (optimalSymbols < requestedSymbols) {
      // Keep top symbols by market cap priority
      const priorityOrder = ['BTC', 'ETH', 'SOL', 'XRP', 'LINK', 'ADA'];
      const newSymbols = this.config.symbols
        .sort((a, b) => priorityOrder.indexOf(a) - priorityOrder.indexOf(b))
        .slice(0, optimalSymbols);

      console.log(`[AutoScale] Reducing symbols: ${this.config.symbols.join(',')} → ${newSymbols.join(',')}`);
      this.config.symbols = newSymbols;

      // Remove states for dropped symbols
      for (const [symbol] of this.states) {
        if (!newSymbols.includes(symbol)) {
          this.states.delete(symbol);
          this.priceHistory.delete(symbol);
        }
      }
    }

    if (optimalLevels !== requestedLevels) {
      console.log(`[AutoScale] Reducing grid levels: ${requestedLevels} → ${optimalLevels}`);
      this.config.gridLevels = optimalLevels;
    }

    if (optimalPositionSize !== requestedPositionSize) {
      console.log(`[AutoScale] Adjusting position size: $${requestedPositionSize} → $${optimalPositionSize}`);
      this.config.positionPerGrid = optimalPositionSize;
    }

    // Recalculate capital per symbol
    const effectiveCapital = this.config.useSimulatedCapital ? this.config.initialCapital : availableEquity;
    this.config.capitalPerSymbol = effectiveCapital / this.config.symbols.length;

    // Update initial capital to match real equity (unless using simulated)
    if (!this.config.useSimulatedCapital) {
      this.config.initialCapital = availableEquity;
      this.initialCapital = availableEquity;
    } else {
      console.log(`[AutoScale] Using SIMULATED capital: $${this.config.initialCapital} (real: $${availableEquity})`);
    }

    // Log final config
    const finalMargin = this.config.symbols.length * this.config.gridLevels * this.config.positionPerGrid;
    const marginUsagePercent = (finalMargin / availableEquity) * 100;

    console.log(`[AutoScale] ====================================`);
    console.log(`[AutoScale] FINAL CONFIG:`);
    console.log(`[AutoScale] - Symbols: ${this.config.symbols.join(', ')} (${this.config.symbols.length})`);
    console.log(`[AutoScale] - Grid Levels: ${this.config.gridLevels}`);
    console.log(`[AutoScale] - Position Size: $${this.config.positionPerGrid}`);
    console.log(`[AutoScale] - Leverage: ${leverage}x`);
    console.log(`[AutoScale] - Capital per Symbol: $${this.config.capitalPerSymbol?.toFixed(2)}`);
    console.log(`[AutoScale] - Margin Required: $${finalMargin.toFixed(2)} (${marginUsagePercent.toFixed(1)}% of equity)`);
    console.log(`[AutoScale] - Estimated Orders: ${this.config.symbols.length * this.config.gridLevels}`);
    console.log(`[AutoScale] ====================================`);
  }

  private createInitialState(symbol: string): GridState {
    return {
      symbol,
      basePrice: 0,
      currentPrice: 0,
      gridLevels: [],
      positions: [],
      trades: [],
      equity: this.config.capitalPerSymbol!,
      realizedPnL: 0,
      unrealizedPnL: 0,
      maxDrawdown: 0,
      totalFees: 0,
      isPaused: false,
      pauseReason: null
    };
  }

  // ============= API METHODS =============

  // ============= MODE HELPERS =============

  /**
   * Returns true if we should use real Kraken API (demo or live)
   * Paper mode uses local simulation only
   */
  private isRealTradingMode(): boolean {
    return this.config.mode === 'demo' || this.config.mode === 'live';
  }

  /**
   * Returns true if this is test mode (local simulation only)
   */
  private isTestMode(): boolean {
    return this.config.mode === 'test';
  }

  /**
   * Get the correct API key for current mode
   */
  private getApiKey(): string {
    if (this.config.mode === 'live') {
      return this.config.liveApiKey || '';
    }
    return this.config.apiKey || '';
  }

  /**
   * Get the correct API secret for current mode
   */
  private getApiSecret(): string {
    if (this.config.mode === 'live') {
      return this.config.liveApiSecret || '';
    }
    return this.config.apiSecret || '';
  }

  private getApiUrl(): string {
    // Paper mode still uses demo URL for price feeds only
    return this.config.mode === 'live' ? this.API_URL_LIVE : this.API_URL_DEMO;
  }

  private getWsUrl(): string {
    // ALWAYS use LIVE WebSocket for real-time prices
    // Demo WebSocket has static/delayed prices which breaks grid trading
    // Orders still go to correct API (demo or live) based on mode
    return this.WS_URL_LIVE;
  }

  private sign(endpoint: string, data: string = '', nonce: string): string {
    // Correct signature per Kraken Futures docs:
    // Step 1: SHA256(postData + nonce + endpoint) - use /api/v3 path (not /derivatives/api/v3)
    // Step 2: HMAC-SHA512(hash) with base64-decoded secret
    // Step 3: Base64-encode result
    const fullEndpoint = '/api/v3' + endpoint;
    const message = data + nonce + fullEndpoint;
    const hash = crypto.createHash('sha256').update(message).digest();
    const apiSecret = this.getApiSecret();
    const signature = crypto.createHmac('sha512', Buffer.from(apiSecret, 'base64'))
      .update(hash)
      .digest('base64');
    return signature;
  }

  private async apiRequest(method: 'GET' | 'POST', endpoint: string, data: Record<string, any> = {}): Promise<any> {
    // Paper mode should not make API calls (except for public endpoints)
    if (this.isTestMode()) {
      throw new Error('API calls not allowed in test mode');
    }

    // Use large nonce to avoid "nonceBelowThreshold" error
    const nonce = (BigInt(Date.now()) * BigInt(1000000) + BigInt(Math.floor(Math.random() * 999999))).toString();
    const postData = method === 'POST' ? new URLSearchParams(data).toString() : '';
    const signature = this.sign(endpoint, postData, nonce);

    const headers = {
      'Content-Type': 'application/x-www-form-urlencoded',
      'APIKey': this.getApiKey(),
      'Nonce': nonce,
      'Authent': signature,
    };

    const url = `${this.getApiUrl()}${endpoint}`;

    try {
      const response = await axios({
        method,
        url: method === 'GET' && Object.keys(data).length > 0
          ? `${url}?${new URLSearchParams(data).toString()}`
          : url,
        data: method === 'POST' ? postData : undefined,
        headers,
        timeout: 10000,
      });
      return response.data;
    } catch (error: any) {
      console.error(`[GridEngine] API Error: ${error.message}`);
      throw error;
    }
  }

  // ============= WEBSOCKET =============

  private async connectWebSocket(): Promise<void> {
    return new Promise((resolve, reject) => {
      const wsUrl = this.getWsUrl();
      console.log(`[GridEngine] Connecting to ${wsUrl}`);

      this.ws = new WebSocket(wsUrl);

      this.ws.on('open', () => {
        console.log('[GridEngine] WebSocket connected');
        this.reconnectAttempts = 0;
        this.subscribeToTickers();
        this.startHeartbeat();
        resolve();
      });

      this.ws.on('message', (data: WebSocket.Data) => {
        this.handleMessage(JSON.parse(data.toString()));
      });

      this.ws.on('close', () => {
        console.log('[GridEngine] WebSocket closed');
        this.stopHeartbeat();
        if (this.isRunning) {
          this.attemptReconnect();
        }
      });

      this.ws.on('error', (error) => {
        console.error('[GridEngine] WebSocket error:', error);
        reject(error);
      });
    });
  }

  private subscribeToTickers(): void {
    if (!this.ws) return;

    const krakenSymbols = this.config.symbols.map(s => getKrakenSymbol(s));

    const subscribeMsg = {
      event: 'subscribe',
      feed: 'ticker',
      product_ids: krakenSymbols
    };

    this.ws.send(JSON.stringify(subscribeMsg));
    console.log(`[GridEngine] Subscribed to tickers: ${krakenSymbols.join(', ')}`);
  }

  private handleMessage(msg: any): void {
    if (msg.feed === 'ticker' && msg.product_id) {
      this.handleTicker(msg);
    } else if (msg.feed === 'fills') {
      this.handleFill(msg);
    } else if (msg.event === 'subscribed') {
      console.log(`[GridEngine] Subscribed to ${msg.feed}`);
    }
  }

  private handleTicker(msg: any): void {
    const symbol = this.getBaseSymbol(msg.product_id);
    if (!symbol || !this.config.symbols.includes(symbol)) return;

    const ticker: KrakenTicker = {
      symbol,
      bid: parseFloat(msg.bid) || 0,
      ask: parseFloat(msg.ask) || 0,
      last: parseFloat(msg.last) || 0,
      volume: parseFloat(msg.volume) || 0,
      change24h: parseFloat(msg.change) || 0
    };

    this.tickers.set(symbol, ticker);

    // Process grid logic
    this.processTickerUpdate(symbol, ticker);
  }

  private handleFill(msg: any): void {
    // Handle order fills
    console.log('[GridEngine] Fill received:', msg);
    this.emit('fill', msg);
  }

  private getBaseSymbol(krakenSymbol: string): string {
    // PF_XBTUSD -> BTC, PF_ETHUSD -> ETH
    const match = krakenSymbol.match(/^PF_(\w+)USD$/);
    if (!match) return '';
    const base = match[1];
    return base === 'XBT' ? 'BTC' : base;
  }

  private startHeartbeat(): void {
    this.heartbeatInterval = setInterval(() => {
      if (this.ws && this.ws.readyState === WebSocket.OPEN) {
        this.ws.ping();
      }
    }, 30000);
  }

  private stopHeartbeat(): void {
    if (this.heartbeatInterval) {
      clearInterval(this.heartbeatInterval);
      this.heartbeatInterval = null;
    }
  }

  private async attemptReconnect(): Promise<void> {
    if (this.reconnectAttempts >= 10) {
      console.error('[GridEngine] Max reconnect attempts reached');
      this.emit('error', new Error('Max reconnect attempts reached'));
      return;
    }

    this.reconnectAttempts++;
    const delay = Math.min(1000 * Math.pow(2, this.reconnectAttempts), 30000);
    console.log(`[GridEngine] Reconnecting in ${delay}ms (attempt ${this.reconnectAttempts})`);

    await new Promise(r => setTimeout(r, delay));

    try {
      await this.connectWebSocket();
    } catch (error) {
      console.error('[GridEngine] Reconnect failed:', error);
    }
  }

  // ============= GRID LOGIC =============

  private setupGrid(symbol: string, price: number): void {
    const state = this.states.get(symbol);
    if (!state) return;

    // Round base price to tick size
    const roundedBasePrice = roundPrice(symbol, price);
    state.basePrice = roundedBasePrice;
    state.currentPrice = price;
    state.gridLevels = [];

    // LONG levels (below base price) - buy when price drops
    for (let i = 1; i <= this.config.gridLevels; i++) {
      const rawGridPrice = roundedBasePrice * (1 - i * this.config.gridSpacing / 100);
      const gridPrice = roundPrice(symbol, rawGridPrice);
      state.gridLevels.push({
        price: gridPrice,
        level: -i,
        isBuyLevel: true,
        side: 'LONG',
        filled: false
      });
    }

    // Determine if SHORTs should be enabled for this symbol
    const shortsEnabled = this.shouldEnableShortsForSymbol(symbol);

    // SHORT levels (above base price) - sell when price rises - ONLY IF ENABLED
    if (shortsEnabled) {
      for (let i = 1; i <= this.config.gridLevels; i++) {
        const rawGridPrice = roundedBasePrice * (1 + i * this.config.gridSpacing / 100);
        const gridPrice = roundPrice(symbol, rawGridPrice);
        state.gridLevels.push({
          price: gridPrice,
          level: i,
          isBuyLevel: false,  // SHORT entry = sell
          side: 'SHORT',
          filled: false
        });
      }
    }

    const biState = this.symbolBidirectionalState.get(symbol);
    const mode = shortsEnabled ? 'BIDIRECTIONAL' : 'LONG-only';
    const reason = biState?.reason || (this.config.bidirectional ? 'manual' : 'disabled');
    console.log(`[GridEngine] Grid setup for ${symbol}: base=${roundedBasePrice}, levels=${state.gridLevels.length} (${mode}) - ${reason}`);
    this.emit('grid_setup', { symbol, basePrice: roundedBasePrice, levels: state.gridLevels.length, bidirectional: shortsEnabled, reason });

    // For DEMO/LIVE mode: Place limit orders at all levels immediately
    if (this.isRealTradingMode()) {
      this.placeGridOrders(symbol);
    }
  }

  // ============= DYNAMIC GRID =============

  /**
   * Check if grid needs to be rebalanced (price moved too far from base)
   * Returns true if rebalance is needed
   */
  private shouldRebalanceGrid(symbol: string): boolean {
    if (!this.config.dynamicGrid) return false;

    const state = this.states.get(symbol);
    if (!state || state.basePrice === 0 || state.currentPrice === 0) return false;

    // Check minimum interval since last rebalance
    const lastRebalance = this.lastGridRebalance.get(symbol) || 0;
    if (Date.now() - lastRebalance < this.MIN_REBALANCE_INTERVAL) return false;

    // Calculate price deviation from base
    const deviation = Math.abs((state.currentPrice - state.basePrice) / state.basePrice * 100);
    const threshold = this.config.gridRebalanceThreshold || 2;

    return deviation >= threshold;
  }

  /**
   * Cancel all unfilled grid orders for a symbol
   */
  private async cancelUnfilledGridOrders(symbol: string): Promise<number> {
    const state = this.states.get(symbol);
    if (!state) return 0;

    let cancelledCount = 0;
    const krakenSymbol = getKrakenSymbol(symbol);

    // Cancel orders from pendingOrders map
    for (const [orderId, order] of this.pendingOrders) {
      if (order.symbol === symbol || order.symbol === krakenSymbol) {
        try {
          const result = await this.cancelOrder(orderId);
          if (result.success) {
            cancelledCount++;
            this.pendingOrders.delete(orderId);
          }
        } catch (error) {
          console.error(`[DynamicGrid] Failed to cancel order ${orderId}:`, error);
        }
      }
    }

    // Also cancel orders from grid levels that have orderIds
    for (const level of state.gridLevels) {
      if (level.orderId && !level.filled) {
        try {
          const result = await this.cancelOrder(level.orderId);
          if (result.success) {
            cancelledCount++;
            level.orderId = undefined;
            level.orderStatus = 'cancelled';
          }
        } catch (error) {
          console.error(`[DynamicGrid] Failed to cancel level order ${level.orderId}:`, error);
        }
      }
    }

    return cancelledCount;
  }

  /**
   * Perform dynamic grid rebalance - recenter grid around current price
   */
  private async rebalanceDynamicGrid(symbol: string): Promise<void> {
    const state = this.states.get(symbol);
    if (!state || !state.currentPrice) return;

    const oldBase = state.basePrice;
    const newBase = state.currentPrice;
    const deviation = ((newBase - oldBase) / oldBase * 100).toFixed(2);

    console.log(`[DynamicGrid] ====================================`);
    console.log(`[DynamicGrid] Rebalancing ${symbol} grid`);
    console.log(`[DynamicGrid] Old base: $${oldBase.toFixed(2)}`);
    console.log(`[DynamicGrid] New base: $${newBase.toFixed(2)} (${deviation}% move)`);
    console.log(`[DynamicGrid] ====================================`);

    // Mark rebalance time
    this.lastGridRebalance.set(symbol, Date.now());

    // Cancel all unfilled orders first
    if (this.isRealTradingMode()) {
      const cancelled = await this.cancelUnfilledGridOrders(symbol);
      console.log(`[DynamicGrid] Cancelled ${cancelled} unfilled orders`);
    }

    // Recenter the grid around current price
    // Note: Existing positions remain with their TP orders
    this.setupGrid(symbol, newBase);

    this.emit('grid_rebalanced', {
      symbol,
      oldBase,
      newBase,
      deviation: parseFloat(deviation),
      timestamp: new Date()
    });
  }

  /**
   * Check and trigger dynamic grid rebalance if needed
   */
  private async checkDynamicGridRebalance(symbol: string): Promise<void> {
    if (this.shouldRebalanceGrid(symbol)) {
      await this.rebalanceDynamicGrid(symbol);
    }
  }

  // ============= HYPER-DYNAMIC GRID SYSTEM =============

  /**
   * Update OHLC candle with new tick price
   */
  private updateOHLC(symbol: string, price: number): void {
    const now = Date.now();
    let candle = this.currentCandle.get(symbol);

    // Start new candle if needed
    if (!candle || now - candle.startTime >= this.CANDLE_INTERVAL_MS) {
      // Save completed candle
      if (candle) {
        const ohlc = this.ohlcData.get(symbol) || [];
        ohlc.push({
          open: candle.open,
          high: candle.high,
          low: candle.low,
          close: candle.close,
          time: candle.startTime
        });
        // Keep last 100 candles for ATR calculation
        if (ohlc.length > 100) ohlc.shift();
        this.ohlcData.set(symbol, ohlc);

        // Recalculate ATR when new candle closes
        this.calculateATRForSymbol(symbol);
      } else {
        // First candle - set default ATR estimate immediately
        this.currentATR.set(symbol, price * 0.005);
        this.currentATRPercent.set(symbol, 0.5);
        console.log(`[HyperDynamic] Initial ATR estimate: 0.5% (no candles yet)`);
      }

      // Start new candle
      candle = {
        open: price,
        high: price,
        low: price,
        close: price,
        startTime: now
      };
    } else {
      // Update current candle
      candle.high = Math.max(candle.high, price);
      candle.low = Math.min(candle.low, price);
      candle.close = price;
    }

    this.currentCandle.set(symbol, candle);
  }

  /**
   * Calculate ATR (Average True Range) from OHLC data
   */
  private calculateATRForSymbol(symbol: string): void {
    const ohlc = this.ohlcData.get(symbol) || [];
    const period = 14;

    if (ohlc.length < period + 1) {
      // Not enough data, estimate ATR from price range
      const state = this.states.get(symbol);
      if (state && state.currentPrice > 0) {
        // Default estimate: 0.5% of price
        this.currentATR.set(symbol, state.currentPrice * 0.005);
        this.currentATRPercent.set(symbol, 0.5);
      }
      return;
    }

    // Calculate True Range for each candle
    const trueRanges: number[] = [];
    for (let i = 1; i < ohlc.length; i++) {
      const current = ohlc[i];
      const prev = ohlc[i - 1];
      const tr = Math.max(
        current.high - current.low,
        Math.abs(current.high - prev.close),
        Math.abs(current.low - prev.close)
      );
      trueRanges.push(tr);
    }

    // Calculate ATR as SMA of True Ranges
    const recentTR = trueRanges.slice(-period);
    const atr = recentTR.reduce((sum, tr) => sum + tr, 0) / recentTR.length;

    const state = this.states.get(symbol);
    const currentPrice = state?.currentPrice || ohlc[ohlc.length - 1].close;
    const atrPercent = (atr / currentPrice) * 100;

    this.currentATR.set(symbol, atr);
    this.currentATRPercent.set(symbol, atrPercent);

    if (this.config.hyperDynamic) {
      console.log(`[HyperDynamic] ${symbol} ATR: $${atr.toFixed(5)} (${atrPercent.toFixed(3)}%)`);
    }
  }

  /**
   * Get dynamic grid spacing based on current ATR
   */
  private getHyperDynamicSpacing(symbol: string): number {
    if (!this.config.hyperDynamic) {
      return this.config.gridSpacing;
    }

    const atrPercent = this.currentATRPercent.get(symbol) || 0.5;
    const multiplier = this.config.atrMultiplier || 0.5;

    // Calculate spacing: ATR% × multiplier
    let spacing = atrPercent * multiplier;

    // Clamp to min/max bounds
    const minSpacing = this.config.minGridSpacing || 0.15;
    const maxSpacing = this.config.maxGridSpacing || 1.5;
    spacing = Math.max(minSpacing, Math.min(maxSpacing, spacing));

    return spacing;
  }

  /**
   * Check if market volatility is above threshold
   */
  private isVolatilitySufficient(symbol: string): boolean {
    if (!this.config.hyperDynamic) return true;

    const atrPercent = this.currentATRPercent.get(symbol) || 0;
    const threshold = this.config.volatilityThreshold || 0.3;

    return atrPercent >= threshold;
  }

  /**
   * Check if hyper-dynamic recalibration is needed
   */
  private shouldHyperRecalibrate(symbol: string): boolean {
    if (!this.config.hyperDynamic || !this.config.tickRecalibration) return false;

    const lastRecal = this.lastHyperRecalibration.get(symbol) || 0;
    if (Date.now() - lastRecal < this.MIN_HYPER_RECALIBRATION_INTERVAL) return false;

    // Check if spacing has changed significantly (>10% change)
    const currentSpacing = this.config.gridSpacing;
    const newSpacing = this.getHyperDynamicSpacing(symbol);
    const changePercent = Math.abs(newSpacing - currentSpacing) / currentSpacing * 100;

    return changePercent > 10;
  }

  /**
   * Perform hyper-dynamic grid recalibration
   */
  private async hyperRecalibrateGrid(symbol: string): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    const oldSpacing = this.config.gridSpacing;
    const newSpacing = this.getHyperDynamicSpacing(symbol);
    const atrPercent = this.currentATRPercent.get(symbol) || 0;

    console.log(`[HyperDynamic] ====================================`);
    console.log(`[HyperDynamic] Recalibrating ${symbol} grid`);
    console.log(`[HyperDynamic] ATR: ${atrPercent.toFixed(3)}%`);
    console.log(`[HyperDynamic] Old spacing: ${oldSpacing.toFixed(3)}%`);
    console.log(`[HyperDynamic] New spacing: ${newSpacing.toFixed(3)}%`);
    console.log(`[HyperDynamic] ====================================`);

    // Update config with new spacing
    this.config.gridSpacing = newSpacing;
    this.lastHyperRecalibration.set(symbol, Date.now());

    // Cancel unfilled orders and rebuild grid
    if (this.isRealTradingMode()) {
      const cancelled = await this.cancelUnfilledGridOrders(symbol);
      console.log(`[HyperDynamic] Cancelled ${cancelled} orders, rebuilding grid`);
    }

    // Rebuild grid with new spacing
    this.setupGrid(symbol, state.currentPrice);

    this.emit('hyper_recalibration', {
      symbol,
      oldSpacing,
      newSpacing,
      atrPercent,
      timestamp: new Date()
    });
  }

  /**
   * Check hyper-dynamic conditions and recalibrate if needed
   */
  private async checkHyperDynamic(symbol: string): Promise<void> {
    if (!this.config.hyperDynamic) return;

    // Check volatility threshold - pause if too low
    if (!this.isVolatilitySufficient(symbol)) {
      const state = this.states.get(symbol);
      if (state && !state.isPaused) {
        const atrPercent = this.currentATRPercent.get(symbol) || 0;
        console.log(`[HyperDynamic] Pausing ${symbol}: ATR ${atrPercent.toFixed(3)}% below threshold ${this.config.volatilityThreshold}%`);
        state.isPaused = true;
        state.pauseReason = `Low volatility: ATR ${atrPercent.toFixed(2)}%`;
        this.emit('trading_paused', { symbol, reason: state.pauseReason });
      }
      return;
    }

    // Unpause if volatility recovered
    const state = this.states.get(symbol);
    if (state?.isPaused && state.pauseReason?.includes('volatility')) {
      console.log(`[HyperDynamic] Resuming ${symbol}: volatility recovered`);
      state.isPaused = false;
      state.pauseReason = null;
      this.emit('trading_resumed', { symbol });
    }

    // Check if recalibration is needed
    if (this.shouldHyperRecalibrate(symbol)) {
      await this.hyperRecalibrateGrid(symbol);
    }
  }

  private async placeGridOrders(symbol: string): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    const mode = this.config.bidirectional ? 'BIDIRECTIONAL' : 'LONG-only';
    console.log(`[GridEngine] Placing limit orders for ${symbol} grid (${mode})...`);

    // IMMEDIATE ENTRY: Place market order to open position immediately at current price
    if (this.config.immediateEntry && state.positions.length === 0) {
      console.log(`[GridEngine] IMMEDIATE ENTRY enabled - opening LONG position at market price`);
      try {
        const symbolInfo = getSymbol(symbol);
        if (symbolInfo) {
          const margin = this.config.positionPerGrid;
          const notional = margin * this.config.leverage;
          const size = roundSize(symbol, notional / state.currentPrice);

          const result = await this.placeMarketOrder(symbol, 'buy', size, false);
          if (result.success) {
            console.log(`[GridEngine] IMMEDIATE ENTRY: LONG opened at market, orderId: ${result.orderId}`);

            // Create position tracking
            const tpPrice = state.currentPrice * (1 + this.config.gridSpacing / 100);
            const position: GridPosition = {
              id: `IMM-${Date.now()}`,
              symbol,
              side: 'LONG',
              entryPrice: state.currentPrice,
              size,
              margin,
              gridLevel: 0,
              tpPrice,
              entryFee: state.currentPrice * size * (symbolInfo.takerFee || 0.0005),
              entryTime: new Date()
            };
            state.positions.push(position);
            this.emit('position_opened', position);
          } else {
            console.error(`[GridEngine] IMMEDIATE ENTRY failed: ${result.error}`);
          }
        }
      } catch (error) {
        console.error(`[GridEngine] IMMEDIATE ENTRY error:`, error);
      }
      await new Promise(resolve => setTimeout(resolve, 300));
    }

    for (const grid of state.gridLevels) {
      // Place orders for all unfilled levels (both LONG and SHORT)
      if (!grid.filled && !grid.orderId) {
        try {
          await this.openPosition(symbol, grid);
          // Small delay to avoid rate limiting
          await new Promise(resolve => setTimeout(resolve, 200));
        } catch (error) {
          console.error(`[GridEngine] Failed to place ${grid.side} order at level ${grid.level}:`, error);
        }
      }
    }
  }

  // ============= AUTO-BIDIRECTIONAL LOGIC =============

  /**
   * Determines if SHORTs should be enabled for a given symbol
   * Uses auto-bidirectional analysis if enabled, otherwise uses config
   */
  private shouldEnableShortsForSymbol(symbol: string): boolean {
    // If autoBidirectional is enabled, use dynamic state
    if (this.config.autoBidirectional) {
      const biState = this.symbolBidirectionalState.get(symbol);
      if (biState) {
        return biState.shortsEnabled;
      }
      // Default to enabled if no state yet (will be analyzed shortly)
      return true;
    }
    // Otherwise use static config
    return this.config.bidirectional ?? false;
  }

  /**
   * Analyzes market trend and determines if SHORTs should be enabled
   * Returns: bullish (disable shorts), bearish (enable shorts only), neutral (both)
   */
  private analyzeTrendForBidirectional(symbol: string): SymbolBidirectionalState {
    const history = this.priceHistory.get(symbol) || [];
    const currentPrice = this.states.get(symbol)?.currentPrice || 0;

    // Need at least 20 data points for analysis
    if (history.length < 20) {
      return {
        shortsEnabled: true,
        longsEnabled: true,
        currentTrend: 'neutral',
        trendStrength: 0,
        lastTrendCheck: Date.now(),
        reason: 'Insufficient data - both enabled'
      };
    }

    // Calculate short-term and medium-term trend
    const shortTermPrices = history.slice(-10);
    const mediumTermPrices = history.slice(-30);

    const shortTermAvg = shortTermPrices.reduce((a, b) => a + b, 0) / shortTermPrices.length;
    const mediumTermAvg = mediumTermPrices.reduce((a, b) => a + b, 0) / mediumTermPrices.length;

    // Calculate momentum (% change over recent period)
    const oldPrice = history[Math.max(0, history.length - 20)];
    const momentum = ((currentPrice - oldPrice) / oldPrice) * 100;

    // Calculate trend strength (0-100)
    const trendStrength = Math.min(100, Math.abs(momentum) * 10);

    // Determine trend direction
    let currentTrend: 'bullish' | 'bearish' | 'neutral';
    let shortsEnabled = true;
    let longsEnabled = true;
    let reason = '';

    // Strong bullish: price significantly above averages AND positive momentum > 2%
    if (shortTermAvg > mediumTermAvg * 1.01 && momentum > 2) {
      currentTrend = 'bullish';
      shortsEnabled = false;  // Disable shorts in bullish market
      longsEnabled = true;
      reason = `🟢 Bullish trend (+${momentum.toFixed(1)}%) - SHORTs disabled`;
    }
    // Strong bearish: price significantly below averages AND negative momentum < -2%
    else if (shortTermAvg < mediumTermAvg * 0.99 && momentum < -2) {
      currentTrend = 'bearish';
      shortsEnabled = true;   // Enable shorts in bearish market
      longsEnabled = false;   // Disable longs in bearish market
      reason = `🔴 Bearish trend (${momentum.toFixed(1)}%) - LONGs disabled`;
    }
    // Neutral/sideways: enable both
    else {
      currentTrend = 'neutral';
      shortsEnabled = true;
      longsEnabled = true;
      reason = `⚪ Neutral trend (${momentum.toFixed(1)}%) - Both enabled`;
    }

    return {
      shortsEnabled,
      longsEnabled,
      currentTrend,
      trendStrength,
      lastTrendCheck: Date.now(),
      reason
    };
  }

  /**
   * Updates bidirectional state for a symbol and manages orders accordingly
   */
  private async updateBidirectionalStateForSymbol(symbol: string): Promise<void> {
    if (!this.config.autoBidirectional) return;

    const oldState = this.symbolBidirectionalState.get(symbol);
    const newState = this.analyzeTrendForBidirectional(symbol);

    // Store new state
    this.symbolBidirectionalState.set(symbol, newState);

    // Check if state changed
    const shortsChanged = oldState?.shortsEnabled !== newState.shortsEnabled;
    const longsChanged = oldState?.longsEnabled !== newState.longsEnabled;

    if (shortsChanged || longsChanged) {
      console.log(`[AutoBidirectional] ${symbol}: ${newState.reason}`);
      this.emit('bidirectional_change', { symbol, ...newState });

      // Handle order changes for real trading modes
      if (this.isRealTradingMode()) {
        if (shortsChanged) {
          if (newState.shortsEnabled) {
            await this.placeShortOrders(symbol);
          } else {
            await this.cancelShortOrders(symbol);
          }
        }
        // Note: We don't cancel LONG orders when longsEnabled is false,
        // we just don't open new ones. Existing LONG positions should run.
      }
    }
  }

  /**
   * Cancels all SHORT orders for a symbol (when disabling shorts)
   */
  private async cancelShortOrders(symbol: string): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    console.log(`[AutoBidirectional] ${symbol}: Cancelling SHORT orders...`);

    let cancelledCount = 0;
    for (const grid of state.gridLevels) {
      if (grid.side === 'SHORT' && grid.orderId && !grid.filled) {
        try {
          await this.cancelKrakenOrder(grid.orderId);
          grid.orderId = undefined;
          grid.orderStatus = undefined;
          cancelledCount++;
        } catch (error) {
          console.error(`[AutoBidirectional] Failed to cancel SHORT order:`, error);
        }
      }
    }

    // Remove SHORT levels from grid
    state.gridLevels = state.gridLevels.filter(g => g.side !== 'SHORT');

    console.log(`[AutoBidirectional] ${symbol}: Cancelled ${cancelledCount} SHORT orders`);
  }

  /**
   * Places SHORT orders for a symbol (when enabling shorts)
   */
  private async placeShortOrders(symbol: string): Promise<void> {
    const state = this.states.get(symbol);
    if (!state || !state.basePrice) return;

    console.log(`[AutoBidirectional] ${symbol}: Placing SHORT orders...`);

    // Add SHORT levels to grid
    let placedCount = 0;
    for (let i = 1; i <= this.config.gridLevels; i++) {
      const rawGridPrice = state.basePrice * (1 + i * this.config.gridSpacing / 100);
      const gridPrice = roundPrice(symbol, rawGridPrice);

      // Check if level already exists
      const existingLevel = state.gridLevels.find(g => g.level === i && g.side === 'SHORT');
      if (existingLevel) continue;

      const newLevel: GridLevel = {
        price: gridPrice,
        level: i,
        isBuyLevel: false,
        side: 'SHORT',
        filled: false
      };

      state.gridLevels.push(newLevel);

      // Place order
      try {
        await this.openPosition(symbol, newLevel);
        placedCount++;
        await new Promise(resolve => setTimeout(resolve, 200));
      } catch (error) {
        console.error(`[AutoBidirectional] Failed to place SHORT order:`, error);
      }
    }

    console.log(`[AutoBidirectional] ${symbol}: Placed ${placedCount} SHORT orders`);
  }

  /**
   * Periodic check for auto-bidirectional - runs every 5 minutes
   */
  private startAutoBidirectionalMonitor(): void {
    if (!this.config.autoBidirectional) return;

    console.log('[AutoBidirectional] Starting trend monitor (checks every 5 min)');

    // Check immediately
    for (const symbol of this.config.symbols) {
      this.updateBidirectionalStateForSymbol(symbol);
    }

    // Then check every 5 minutes
    this.autoBidirectionalInterval = setInterval(() => {
      for (const symbol of this.config.symbols) {
        this.updateBidirectionalStateForSymbol(symbol);
      }
    }, 5 * 60 * 1000);
  }

  /**
   * Stop auto-bidirectional monitor
   */
  private stopAutoBidirectionalMonitor(): void {
    if (this.autoBidirectionalInterval) {
      clearInterval(this.autoBidirectionalInterval);
      this.autoBidirectionalInterval = null;
    }
  }

  private processTickerUpdate(symbol: string, ticker: KrakenTicker): void {
    const state = this.states.get(symbol);
    if (!state) return;

    const price = ticker.last;
    state.currentPrice = price;

    // Track price history for trend filter
    const history = this.priceHistory.get(symbol) || [];
    history.push(price);
    if (history.length > 100) history.shift();
    this.priceHistory.set(symbol, history);

    // HYPER-DYNAMIC: Update OHLC data for ATR calculation
    if (this.config.hyperDynamic) {
      this.updateOHLC(symbol, price);
    }

    // Initialize grid if needed
    if (state.basePrice === 0) {
      // For hyper-dynamic, use ATR-based spacing from start
      if (this.config.hyperDynamic) {
        const dynamicSpacing = this.getHyperDynamicSpacing(symbol);
        this.config.gridSpacing = dynamicSpacing;
        console.log(`[HyperDynamic] Initial spacing: ${dynamicSpacing.toFixed(3)}%`);
      }
      this.setupGrid(symbol, price);
      return;
    }

    // Update equity and drawdown
    this.updateEquity(symbol);

    // HYPER-DYNAMIC: Check volatility threshold and recalibration
    if (this.config.hyperDynamic) {
      this.checkHyperDynamic(symbol).catch(err => {
        console.error(`[HyperDynamic] Check failed for ${symbol}:`, err);
      });
    }

    // Check dynamic grid rebalance (async, non-blocking)
    if (this.config.dynamicGrid && !this.config.hyperDynamic) {
      // Skip standard rebalance if hyper-dynamic is handling it
      this.checkDynamicGridRebalance(symbol).catch(err => {
        console.error(`[DynamicGrid] Rebalance check failed for ${symbol}:`, err);
      });
    }

    // Check circuit breaker
    if (this.checkCircuitBreaker(symbol)) return;

    // Check trend filter
    if (this.config.trendFilterEnabled && this.checkTrendFilter(symbol)) return;

    // Check for take profit
    this.checkTakeProfit(symbol, ticker.ask);

    // Check for new entries
    if (!state.isPaused) {
      this.checkNewEntries(symbol, ticker.bid);
    }

    // Emit update
    this.emit('tick', {
      symbol,
      price,
      equity: state.equity,
      positions: state.positions.length,
      realizedPnL: state.realizedPnL,
      unrealizedPnL: state.unrealizedPnL
    });
  }

  private updateEquity(symbol: string): void {
    const state = this.states.get(symbol);
    if (!state) return;

    let unrealizedPnL = 0;
    for (const pos of state.positions) {
      // LONG profits when price rises, SHORT profits when price drops
      if (pos.side === 'LONG') {
        unrealizedPnL += (state.currentPrice - pos.entryPrice) * pos.size;
      } else {
        unrealizedPnL += (pos.entryPrice - state.currentPrice) * pos.size;
      }
    }

    state.unrealizedPnL = unrealizedPnL;
    const capital = this.config.capitalPerSymbol!;
    state.equity = capital + state.realizedPnL + unrealizedPnL;

    // Update max drawdown
    const dd = ((capital - state.equity) / capital) * 100;
    if (dd > state.maxDrawdown) state.maxDrawdown = dd;
  }

  private checkCircuitBreaker(symbol: string): boolean {
    const state = this.states.get(symbol);
    if (!state) return false;

    const capital = this.config.capitalPerSymbol!;
    const dd = ((capital - state.equity) / capital) * 100;

    if (dd > this.config.maxDrawdownPercent) {
      state.isPaused = true;
      state.pauseReason = `Max drawdown exceeded: ${dd.toFixed(1)}%`;
      this.emit('circuit_breaker', { symbol, drawdown: dd });
      return true;
    }

    return false;
  }

  private checkTrendFilter(symbol: string): boolean {
    const state = this.states.get(symbol);
    const history = this.priceHistory.get(symbol) || [];

    if (history.length < 24) return false;

    const oldPrice = history[history.length - 24];
    const currentPrice = state?.currentPrice || 0;
    const movePercent = Math.abs((currentPrice - oldPrice) / oldPrice * 100);

    if (movePercent > this.config.trendFilterThreshold) {
      if (state) {
        state.isPaused = true;
        state.pauseReason = `Strong trend: ${movePercent.toFixed(1)}% move`;
      }
      return true;
    }

    if (state) {
      state.isPaused = false;
      state.pauseReason = null;
    }
    return false;
  }

  private async checkTakeProfit(symbol: string, askPrice: number): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    if (this.isTestMode()) {
      // TEST MODE: Check and close positions locally when price hits TP
      for (let i = state.positions.length - 1; i >= 0; i--) {
        const pos = state.positions[i];
        if (askPrice >= pos.tpPrice) {
          await this.closePosition(symbol, pos, 'TP');
        }
      }
    } else {
      // DEMO/LIVE MODE: TP orders are already placed on Kraken
      // We just ensure every position has a TP order
      for (const pos of state.positions) {
        if (!pos.tpOrderId) {
          // Position doesn't have TP order yet - place it
          console.log(`[GridEngine] Position ${pos.id} missing TP order, placing now...`);
          await this.placeTakeProfitOrder(pos);
        }
      }
      // TP execution is handled by syncWithKraken() when SELL orders are filled
    }
  }

  private async checkNewEntries(symbol: string, bidPrice: number): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    if (state.positions.length >= this.config.maxOpenPositions) return;

    for (const grid of state.gridLevels) {
      if (grid.isBuyLevel && !grid.filled && bidPrice <= grid.price) {
        await this.openPosition(symbol, grid);

        if (state.positions.length >= this.config.maxOpenPositions) break;
      }
    }
  }

  // ============= ORDER EXECUTION =============

  private async openPosition(symbol: string, grid: GridLevel): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    const symbolInfo = getSymbol(symbol);
    if (!symbolInfo) return;

    // Don't place if already has pending order for this level
    if (grid.orderId && grid.orderStatus === 'pending') {
      console.log(`[GridEngine] Level ${grid.level} already has pending order ${grid.orderId}`);
      return;
    }

    const margin = this.config.positionPerGrid;
    const notional = margin * this.config.leverage;
    const size = roundSize(symbol, notional / grid.price);

    console.log(`[GridEngine] Opening position: ${symbol} size=${size} @ ${grid.price}`);

    if (this.isTestMode()) {
      // TEST MODE: Simulate immediate fill at current price
      await this.simulateOpenPosition(symbol, grid, size);
    } else {
      // DEMO/LIVE MODE: Place real limit order on Kraken
      await this.placeRealOrder(symbol, grid, size);
    }
  }

  /**
   * TEST MODE: Simulate position open with immediate fill
   */
  private async simulateOpenPosition(symbol: string, grid: GridLevel, size: number): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    const symbolInfo = getSymbol(symbol);
    if (!symbolInfo) return;

    const fillPrice = state.currentPrice; // Use current price as fill price
    const fee = fillPrice * size * symbolInfo.makerFee;

    // TP depends on side: LONG profits when price rises, SHORT profits when price drops
    const tpPrice = grid.side === 'LONG'
      ? fillPrice * (1 + this.config.gridSpacing / 100)   // LONG: TP above
      : fillPrice * (1 - this.config.gridSpacing / 100);  // SHORT: TP below

    const position: GridPosition = {
      id: `P${++this.tradeIdCounter}`,
      symbol,
      side: grid.side,
      entryPrice: fillPrice,
      size,
      margin: this.config.positionPerGrid,
      gridLevel: grid.level,
      tpPrice,
      entryFee: fee,
      entryTime: new Date()
    };

    // Mark grid as filled
    grid.filled = true;
    grid.orderStatus = 'filled';

    // Add position
    state.positions.push(position);
    state.totalFees += fee;

    // Update paper balance
    this.testBalance -= fee;

    console.log(`[GridEngine] [PAPER] ${grid.side} position opened: ${symbol} @ ${fillPrice}, size=${size}, TP=${position.tpPrice.toFixed(2)}`);
    this.emit('position_opened', position);
  }

  /**
   * DEMO/LIVE MODE: Place real limit order on Kraken
   */
  private async placeRealOrder(symbol: string, grid: GridLevel, size: number): Promise<void> {
    try {
      // LONG = buy entry, SHORT = sell entry
      const orderSide = grid.side === 'LONG' ? 'buy' : 'sell';
      const orderResult = await this.placeLimitOrder(symbol, orderSide, size, grid.price);

      if (orderResult.success && orderResult.orderId) {
        // Track as pending order - DO NOT add position yet!
        grid.orderId = orderResult.orderId;
        grid.orderStatus = 'pending';

        this.pendingOrders.set(orderResult.orderId, {
          orderId: orderResult.orderId,
          symbol,
          side: orderSide,
          positionSide: grid.side, // LONG or SHORT
          size,
          price: grid.price,
          gridLevel: grid.level,
          placedAt: new Date()
        });

        console.log(`[GridEngine] [${this.config.mode.toUpperCase()}] ${grid.side} LIMIT ORDER PLACED: ${orderResult.orderId}`);
        console.log(`[GridEngine] Waiting for fill... (pending orders: ${this.pendingOrders.size})`);

        this.emit('order_placed', { orderId: orderResult.orderId, symbol, side: orderSide, positionSide: grid.side, size, price: grid.price });
      } else {
        console.error(`[GridEngine] Order rejected by Kraken: ${orderResult.error}`);
      }
    } catch (error) {
      console.error(`[GridEngine] Order execution error:`, error);
    }
  }

  private async closePosition(symbol: string, position: GridPosition, reason: 'TP' | 'SL' | 'CLOSE'): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    const symbolInfo = getSymbol(symbol);
    if (!symbolInfo) return;

    if (this.isTestMode()) {
      // TEST MODE: Simulate close immediately
      await this.simulateClosePosition(symbol, position, reason);
    } else {
      // DEMO/LIVE MODE: Execute real close on Kraken
      // For TP orders in demo/live, they're handled by syncWithKraken when the limit order fills
      if (reason === 'TP') {
        console.log(`[GridEngine] TP orders are handled by Kraken sync, not closePosition`);
        return;
      }

      // Cancel any pending TP order for this position first
      if (position.tpOrderId) {
        try {
          await this.cancelOrder(position.tpOrderId);
          console.log(`[GridEngine] Cancelled TP order ${position.tpOrderId} before manual close`);
          this.pendingOrders.delete(position.tpOrderId);
        } catch (error) {
          console.error(`[GridEngine] Failed to cancel TP order:`, error);
        }
      }

      // Execute market order on Kraken for immediate close
      // LONG closes with sell, SHORT closes with buy
      const closeSide = position.side === 'LONG' ? 'sell' : 'buy';
      try {
        const result = await this.placeMarketOrder(symbol, closeSide, position.size, true);

        if (!result.success) {
          console.error(`[GridEngine] Close order FAILED: ${result.error}`);
          return; // Don't update local state if order failed
        }

        console.log(`[GridEngine] ${position.side} close order (${reason}) on Kraken ${this.config.mode.toUpperCase()}: ${result.orderId}`);

        // Market orders are executed immediately, so we can update state
        this.finalizePositionClose(symbol, position, state.currentPrice, reason);

      } catch (error) {
        console.error(`[GridEngine] Close order error:`, error);
      }
    }
  }

  /**
   * TEST MODE: Simulate position close
   */
  private async simulateClosePosition(symbol: string, position: GridPosition, reason: 'TP' | 'SL' | 'CLOSE'): Promise<void> {
    const state = this.states.get(symbol);
    if (!state) return;

    // In test mode, use TP price for TP, current price for SL/CLOSE
    const exitPrice = reason === 'TP' ? position.tpPrice : state.currentPrice;
    this.finalizePositionClose(symbol, position, exitPrice, reason);

    console.log(`[GridEngine] [PAPER] Position closed (${reason}): ${symbol} @ ${exitPrice}`);
  }

  /**
   * Finalize position close - update state, record trade, emit event
   * Used by both paper and real trading modes
   */
  private finalizePositionClose(symbol: string, position: GridPosition, exitPrice: number, reason: 'TP' | 'SL' | 'CLOSE'): void {
    const state = this.states.get(symbol);
    if (!state) return;

    const symbolInfo = getSymbol(symbol);
    if (!symbolInfo) return;

    // PnL depends on position side
    // LONG: profit when price rises (exit > entry)
    // SHORT: profit when price drops (entry > exit)
    const pnl = position.side === 'LONG'
      ? (exitPrice - position.entryPrice) * position.size
      : (position.entryPrice - exitPrice) * position.size;

    const exitFee = this.isTestMode()
      ? exitPrice * position.size * symbolInfo.makerFee  // Paper uses maker fee (simulating limit)
      : position.margin * this.config.leverage * symbolInfo.takerFee; // Market = taker fee

    const trade: GridTrade = {
      id: `T${++this.tradeIdCounter}`,
      symbol,
      side: position.side,
      entryPrice: position.entryPrice,
      exitPrice,
      pnl,
      fees: position.entryFee + exitFee,
      netPnl: pnl - position.entryFee - exitFee,
      entryTime: position.entryTime,
      exitTime: new Date(),
      result: reason
    };

    state.trades.push(trade);
    state.realizedPnL += trade.netPnl;
    state.totalFees += exitFee;

    // Update paper balance
    if (this.isTestMode()) {
      this.testBalance += trade.netPnl;
    }

    // Reset grid level
    const grid = state.gridLevels.find(g => g.level === position.gridLevel);
    if (grid) {
      grid.filled = false;
      grid.orderId = undefined;
      grid.orderStatus = undefined;
    }

    // Remove position
    const idx = state.positions.indexOf(position);
    if (idx >= 0) state.positions.splice(idx, 1);

    console.log(`[GridEngine] ${position.side} position closed: ${symbol} @ ${exitPrice}, PnL=${trade.netPnl.toFixed(2)}`);
    this.emit('position_closed', trade);
  }

  private async placeLimitOrder(symbol: string, side: 'buy' | 'sell', size: number, price: number): Promise<{ success: boolean; orderId?: string; error?: string }> {
    const symbolInfo = getSymbol(symbol);
    if (!symbolInfo) {
      console.error(`[GridEngine] placeLimitOrder: Unknown symbol ${symbol}`);
      return { success: false, error: 'Unknown symbol' };
    }

    // Ensure price is properly rounded to tick size
    const roundedPrice = roundPrice(symbol, price);
    const roundedSize = roundSize(symbol, size);

    const orderParams = {
      orderType: 'lmt',
      symbol: symbolInfo.krakenSymbol,
      side,
      size: roundedSize.toString(),
      limitPrice: roundedPrice.toString()
    };

    console.log(`[GridEngine] Sending limit order to Kraken:`, JSON.stringify(orderParams));

    try {
      const response = await this.apiRequest('POST', '/sendorder', orderParams);

      console.log(`[GridEngine] Kraken order response:`, JSON.stringify(response).slice(0, 500));

      if (response.result === 'success') {
        const orderId = response.sendStatus?.order_id;
        const status = response.sendStatus?.status;

        // Check if order was actually placed - valid statuses: placed, filled, partiallyFilled
        const validStatuses = ['placed', 'filled', 'partiallyFilled'];
        if (status && validStatuses.includes(status)) {
          console.log(`[GridEngine] Order SUCCESS: ${orderId} status=${status}`);
          return { success: true, orderId };
        }

        // Order was rejected (invalidPrice, insufficientFunds, etc.)
        console.error(`[GridEngine] Order REJECTED: ${orderId} status=${status}`);
        return { success: false, error: `Order rejected: ${status}`, orderId };
      }

      console.error(`[GridEngine] Order FAILED:`, response.error || response);
      return { success: false, error: response.error || 'Order failed' };
    } catch (error: any) {
      console.error(`[GridEngine] Order EXCEPTION:`, error.message);
      return { success: false, error: error.message };
    }
  }

  private async placeMarketOrder(symbol: string, side: 'buy' | 'sell', size: number, reduceOnly: boolean = false): Promise<{ success: boolean; orderId?: string; error?: string }> {
    const symbolInfo = getSymbol(symbol);
    if (!symbolInfo) return { success: false, error: 'Unknown symbol' };

    try {
      const data: Record<string, string> = {
        orderType: 'mkt',
        symbol: symbolInfo.krakenSymbol,
        side,
        size: size.toString()
      };

      if (reduceOnly) data.reduceOnly = 'true';

      const response = await this.apiRequest('POST', '/sendorder', data);

      if (response.result === 'success') {
        return { success: true, orderId: response.sendStatus?.order_id };
      }

      return { success: false, error: response.error || 'Order failed' };
    } catch (error: any) {
      return { success: false, error: error.message };
    }
  }

  private async cancelOrder(orderId: string): Promise<{ success: boolean; error?: string }> {
    try {
      const response = await this.apiRequest('POST', '/cancelorder', { order_id: orderId });

      if (response.result === 'success') {
        console.log(`[GridEngine] Order cancelled: ${orderId}`);
        return { success: true };
      }

      return { success: false, error: response.error || 'Cancel failed' };
    } catch (error: any) {
      return { success: false, error: error.message };
    }
  }

  // ============= PUBLIC API =============

  async start(): Promise<void> {
    if (this.isRunning) return;

    console.log(`[GridEngine] ====================================`);
    console.log(`[GridEngine] Starting in ${this.config.mode.toUpperCase()} mode`);
    console.log(`[GridEngine] ====================================`);

    if (this.isTestMode()) {
      console.log(`[GridEngine] Mode: TEST (backtest with Binance data)`);
      console.log(`[GridEngine] - Data: Binance Historical API (tick-by-tick)`);
      console.log(`[GridEngine] - Orders: Simulated locally (Kraken-identical logic)`);
      console.log(`[GridEngine] - Positions: Tracked locally`);
      console.log(`[GridEngine] - Equity: $${this.config.initialCapital} (simulated)`);
    } else {
      console.log(`[GridEngine] Mode: ${this.config.mode.toUpperCase()} (real Kraken API)`);
      console.log(`[GridEngine] - API: ${this.config.mode === 'live' ? 'futures.kraken.com' : 'demo-futures.kraken.com'}`);
      console.log(`[GridEngine] - Orders: Real orders on Kraken`);
      console.log(`[GridEngine] - Positions: Synced from Kraken API`);
    }

    console.log(`[GridEngine] Symbols: ${this.config.symbols.join(', ')}`);
    console.log(`[GridEngine] Budget: $${this.config.initialCapital}`);
    console.log(`[GridEngine] Grid: ${this.config.gridSpacing}% spacing, ${this.config.gridLevels} levels`);
    console.log(`[GridEngine] Auto Strategy: ${this.config.autoStrategy ? 'ENABLED' : 'disabled'}`);
    console.log(`[GridEngine] Dynamic Grid: ${this.config.dynamicGrid ? `ENABLED (rebalance at ${this.config.gridRebalanceThreshold}% deviation)` : 'disabled'}`);

    // Initialize based on mode
    if (this.isTestMode()) {
      // Paper mode: use local simulation
      this.testBalance = this.config.initialCapital;
      this.krakenEquity = this.config.initialCapital;
      this.krakenAvailableMargin = this.config.initialCapital;
      console.log(`[GridEngine] Test balance initialized: $${this.testBalance}`);
    } else {
      // Demo/Live mode: connect to Kraken API
      try {
        const accountInfo = await this.getKrakenAccountInfo();
        if (!accountInfo.success) {
          throw new Error(`Kraken API connection failed: ${accountInfo.error}`);
        }
        console.log(`[GridEngine] Kraken ${this.config.mode.toUpperCase()} connected!`);
        console.log(`[GridEngine] Account equity: $${accountInfo.equity?.toFixed(2)}`);
        console.log(`[GridEngine] Available margin: $${accountInfo.availableMargin?.toFixed(2)}`);

        // Store initial Kraken balance (or simulated)
        if (this.config.useSimulatedCapital) {
          this.krakenEquity = this.config.initialCapital;
          this.krakenAvailableMargin = this.config.initialCapital;
          console.log(`[GridEngine] Using SIMULATED capital: $${this.config.initialCapital}`);
        } else {
          this.krakenEquity = accountInfo.equity || 0;
          this.krakenAvailableMargin = accountInfo.availableMargin || 0;
        }

        // AUTO-SCALE: Adjust config based on equity
        this.autoScaleConfig(this.krakenEquity);

        // Start periodic balance updates (every 30 seconds)
        this.startBalanceUpdates();

        // Import existing orders from Kraken (orders placed before this session)
        await this.importExistingOrders();

        // Start order sync with Kraken (checks fills every 5 seconds)
        this.startOrderSync();
      } catch (error: any) {
        console.error(`[GridEngine] Failed to connect to Kraken ${this.config.mode}: ${error.message}`);
        throw error;
      }
    }

    this.isRunning = true;
    this.startTime = Date.now();

    try {
      if (this.isTestMode()) {
        // TEST MODE: Run backtest with Binance historical data
        console.log(`[GridEngine] Starting backtest simulation...`);
        console.log(`[GridEngine] - Data source: Binance Historical API`);
        console.log(`[GridEngine] - Period: ${this.config.testStartDate || 'last 24h'} to ${this.config.testEndDate || 'now'}`);
        console.log(`[GridEngine] - Interval: ${this.config.testCandleInterval || '1m'}`);

        // Run backtest asynchronously
        this.runBacktest().catch(err => {
          console.error('[GridEngine] Backtest error:', err);
          this.isRunning = false;
        });
      } else {
        // DEMO/LIVE MODE: Connect to Kraken WebSocket
        await this.connectWebSocket();

        // Start auto strategy analysis if enabled
        if (this.config.autoStrategy) {
          this.startStrategyAnalysis();
        }

        // Start auto-bidirectional monitor if enabled
        if (this.config.autoBidirectional) {
          this.startAutoBidirectionalMonitor();
        }
      }

      this.emit('started', { mode: this.config.mode, symbols: this.config.symbols });
    } catch (error) {
      this.isRunning = false;
      throw error;
    }
  }

  /**
   * Run backtest simulation using Binance historical data
   * This is used in TEST mode to simulate trading tick-by-tick
   */
  private async runBacktest(): Promise<void> {
    const endTime = this.config.testEndDate
      ? new Date(this.config.testEndDate).getTime()
      : Date.now();
    const startTime = this.config.testStartDate
      ? new Date(this.config.testStartDate).getTime()
      : endTime - 24 * 60 * 60 * 1000; // Default: last 24 hours

    const interval = this.config.testCandleInterval || '1m';
    const speed = this.config.testSpeed || 100; // Default 100x speed

    console.log(`[GridEngine] Fetching historical data from Binance...`);

    // Process each symbol
    for (const symbol of this.config.symbols) {
      const binanceSymbol = symbol === 'BTC' ? 'BTCUSDT' : `${symbol}USDT`;

      console.log(`[GridEngine] Fetching ${binanceSymbol} candles...`);

      let candleCount = 0;
      let lastLogTime = Date.now();

      // Fetch and process candles
      for await (const candle of BinanceHistoricalData.fetchKlines(binanceSymbol, interval, startTime, endTime)) {
        if (!this.isRunning) break;

        candleCount++;

        // Create ticker from candle
        const ticker: KrakenTicker = {
          symbol,
          bid: candle.close * 0.9999, // Approximate bid
          ask: candle.close * 1.0001, // Approximate ask
          last: candle.close,
          volume: candle.volume,
          change24h: 0
        };

        this.tickers.set(symbol, ticker);

        // Initialize grid on first candle
        if (candleCount === 1) {
          this.setupGrid(symbol, candle.close);
        }

        // Process ticker update (this handles grid logic, positions, etc.)
        this.processTickerUpdate(symbol, ticker);

        // Emit progress every second
        if (Date.now() - lastLogTime > 1000) {
          const progress = ((candle.timestamp - startTime) / (endTime - startTime)) * 100;
          console.log(`[GridEngine] Backtest progress: ${progress.toFixed(1)}% - ${new Date(candle.timestamp).toISOString()}`);
          this.emit('backtest_progress', {
            symbol,
            progress,
            timestamp: candle.timestamp,
            candles: candleCount
          });
          lastLogTime = Date.now();
        }

        // Speed control (faster simulation)
        if (speed < 1000) {
          await new Promise(r => setTimeout(r, 1000 / speed));
        }
      }

      console.log(`[GridEngine] Processed ${candleCount} candles for ${symbol}`);
    }

    // Backtest complete
    console.log(`[GridEngine] Backtest completed!`);
    console.log(`[GridEngine] Final equity: $${this.testBalance.toFixed(2)}`);

    const state = this.states.get(this.config.symbols[0]);
    if (state) {
      console.log(`[GridEngine] Total trades: ${state.trades.length}`);
      console.log(`[GridEngine] Realized P&L: $${state.realizedPnL.toFixed(2)}`);
    }

    this.emit('backtest_complete', {
      finalEquity: this.testBalance,
      trades: this.config.symbols.map(s => this.states.get(s)?.trades || []).flat(),
      duration: Date.now() - this.startTime
    });

    this.isRunning = false;
  }

  /**
   * Get account info from Kraken API
   *
   * IMPORTANT: This distinguishes between:
   * - BALANCE: Total deposited funds (cash account)
   * - EQUITY: Balance + unrealized PnL (portfolio value)
   * - AVAILABLE MARGIN: Funds available for new positions
   *
   * For trading, EQUITY is what matters (includes open positions PnL)
   */
  async getKrakenAccountInfo(): Promise<{ success: boolean; equity?: number; balance?: number; availableMargin?: number; error?: string }> {
    // Paper mode: return simulated values
    if (this.isTestMode()) {
      return {
        success: true,
        equity: this.testBalance,
        balance: this.testBalance,
        availableMargin: this.testBalance
      };
    }

    try {
      const response = await this.apiRequest('GET', '/accounts');

      if (response.result === 'success' && response.accounts) {
        const accounts = response.accounts;

        let equity = 0;
        let balance = 0;
        let availableMargin = 0;

        // Get current prices for crypto conversion
        const btcPrice = this.tickers.get('BTC')?.last || 95000;
        const ethPrice = this.tickers.get('ETH')?.last || 3300;

        // 1. Check FLEX account (multi-collateral margin) - this is the main trading account
        // portfolioValue = EQUITY (all collateral + unrealized PnL, in USD)
        // availableMargin = funds available for new positions
        if (accounts.flex) {
          const flex = accounts.flex;
          equity = flex.portfolioValue || flex.pv || 0;
          availableMargin = flex.availableMargin || flex.auxiliary?.af || 0;
          balance = equity; // For flex account, equity = balance

          if (equity > 0) {
            console.log(`[GridEngine] Flex account - Equity: $${equity.toFixed(2)}, Available: $${availableMargin.toFixed(2)}`);
            return {
              success: true,
              equity,
              balance,
              availableMargin
            };
          }
        }

        // 2. Fallback: Check fi_xbtusd account (older single-collateral account)
        if (accounts.fi_xbtusd) {
          const fi = accounts.fi_xbtusd;
          equity = fi.portfolioValue || fi.pv || fi.auxiliary?.pv || 0;
          availableMargin = fi.availableMargin || fi.auxiliary?.af || 0;
          if (equity > 0) {
            console.log(`[GridEngine] FI_XBTUSD account - Equity: $${equity.toFixed(2)}`);
            return { success: true, equity, balance: equity, availableMargin };
          }
        }

        // 3. Final fallback: Calculate from CASH account (all deposited funds)
        // This is only used when flex account is truly empty (e.g., no positions ever opened)
        if (accounts.cash?.balances) {
          const balances = accounts.cash.balances;

          // Calculate USD equivalent of all holdings
          balance = (balances.usd || 0) +
                    (balances.usdt || 0) +
                    (balances.usdc || 0) +
                    (balances['usd credit'] || 0) +
                    (balances.eur || 0) * 1.1 +
                    (balances.gbp || 0) * 1.27 +
                    (balances.xbt || 0) * btcPrice +
                    (balances.eth || 0) * ethPrice +
                    (balances.ltc || 0) * 100 +
                    (balances.xrp || 0) * 2.5 +
                    (balances.bch || 0) * 450;

          // For accounts without open positions, equity = balance
          equity = balance;
          availableMargin = balance;

          console.log(`[GridEngine] Cash account balance: $${balance.toFixed(2)} (USD equivalent)`);
        }

        // 3. Check for unrealized PnL from open positions
        // NOTE: Individual margin accounts (fi_xbtusd, fi_xrpusd, etc.) show collateral in their
        // respective currencies, NOT in USD. The pv in these accounts is the collateral amount,
        // not the USD value. We should use the flex account or calculate from positions instead.
        // Skip individual margin account iteration as it leads to incorrect totals.

        if (equity > 0 || balance > 0) {
          return {
            success: true,
            equity: equity || balance,
            balance,
            availableMargin: availableMargin || balance
          };
        }

        console.warn('[GridEngine] No funds found in any Kraken account');
        return { success: false, error: 'No funds found in Kraken account' };
      }

      if (response.result === 'error') {
        console.error('[GridEngine] Kraken API error:', response.error);
        return { success: false, error: response.error || 'API error' };
      }

      return { success: false, error: 'No accounts found' };
    } catch (error: any) {
      console.error('[GridEngine] getKrakenAccountInfo error:', error);
      // Fallback for demo mode
      if (this.config.mode === 'demo' || this.config.mode === 'test') {
        return {
          success: true,
          equity: this.config.initialCapital,
          balance: this.config.initialCapital,
          availableMargin: this.config.initialCapital
        };
      }
      return { success: false, error: error.message };
    }
  }

  stop(): void {
    if (!this.isRunning) return;

    console.log('[GridEngine] Stopping...');
    this.isRunning = false;

    this.stopHeartbeat();
    this.stopStrategyAnalysis();
    this.stopAutoBidirectionalMonitor();
    this.stopBalanceUpdates();
    this.stopOrderSync();

    if (this.ws) {
      this.ws.close();
      this.ws = null;
    }

    this.emit('stopped');
  }

  private startBalanceUpdates(): void {
    // Skip balance updates if using simulated capital
    if (this.config.useSimulatedCapital) {
      console.log('[GridEngine] Using simulated capital - balance updates disabled');
      return;
    }

    // Update balance every 30 seconds
    this.balanceUpdateInterval = setInterval(async () => {
      try {
        const accountInfo = await this.getKrakenAccountInfo();
        if (accountInfo.success) {
          this.krakenEquity = accountInfo.equity || 0;
          this.krakenAvailableMargin = accountInfo.availableMargin || 0;
        }
      } catch (error) {
        console.error('[GridEngine] Balance update error:', error);
      }
    }, 30000);
  }

  private stopBalanceUpdates(): void {
    if (this.balanceUpdateInterval) {
      clearInterval(this.balanceUpdateInterval);
      this.balanceUpdateInterval = null;
    }
  }

  // ============= KRAKEN ORDER SYNC =============

  private startOrderSync(): void {
    console.log('[GridEngine] Starting order sync with Kraken (every 5 seconds)');

    // Sync immediately
    setTimeout(() => this.syncWithKraken(), 2000);

    // Then sync every 5 seconds
    this.orderSyncInterval = setInterval(() => {
      this.syncWithKraken();
    }, 5000);
  }

  private stopOrderSync(): void {
    if (this.orderSyncInterval) {
      clearInterval(this.orderSyncInterval);
      this.orderSyncInterval = null;
    }
  }

  /**
   * Import existing orders from Kraken that were placed before this session.
   * This allows the engine to track and manage orders that survived a restart.
   */
  private async importExistingOrders(): Promise<void> {
    console.log('[GridEngine] Importing existing orders from Kraken...');

    try {
      const openOrdersResponse = await this.apiRequest('GET', '/openorders');

      if (openOrdersResponse.result !== 'success' || !openOrdersResponse.openOrders) {
        console.log('[GridEngine] No existing orders to import');
        return;
      }

      const orders = openOrdersResponse.openOrders;
      console.log(`[GridEngine] Found ${orders.length} existing orders on Kraken`);

      let imported = 0;
      let skipped = 0;

      for (const order of orders) {
        // Only import limit buy orders (grid entries)
        if (order.orderType !== 'lmt' || order.side !== 'buy') {
          skipped++;
          continue;
        }

        // Extract symbol from Kraken format (PF_XBTUSD -> BTC)
        const krakenSymbol = order.symbol;
        let baseSymbol = krakenSymbol.replace('PF_', '').replace('USD', '');
        if (baseSymbol === 'XBT') baseSymbol = 'BTC';

        // Check if this symbol is in our trading list
        if (!this.config.symbols.includes(baseSymbol)) {
          skipped++;
          continue;
        }

        const state = this.states.get(baseSymbol);
        if (!state) {
          skipped++;
          continue;
        }

        const orderPrice = parseFloat(order.limitPrice);
        const orderSize = parseFloat(order.unfilledSize || order.quantity);

        // Find matching grid level by price (within 0.1% tolerance)
        const matchingGrid = state.gridLevels.find(g => {
          const priceDiff = Math.abs(g.price - orderPrice) / orderPrice;
          return priceDiff < 0.001 && g.isBuyLevel;
        });

        if (matchingGrid) {
          // Mark grid as having an order
          matchingGrid.orderId = order.order_id;
          matchingGrid.orderStatus = 'pending';
          matchingGrid.filled = false;

          // Add to pending orders for tracking
          this.pendingOrders.set(order.order_id, {
            orderId: order.order_id,
            symbol: baseSymbol,
            side: 'buy',
            size: orderSize,
            price: orderPrice,
            gridLevel: matchingGrid.level,
            placedAt: new Date(order.receivedTime || Date.now())
          });

          imported++;
          console.log(`[GridEngine] Imported order: ${baseSymbol} BUY @ $${orderPrice} (level ${matchingGrid.level})`);
        } else {
          // Order doesn't match any grid level - still track it
          this.pendingOrders.set(order.order_id, {
            orderId: order.order_id,
            symbol: baseSymbol,
            side: 'buy',
            size: orderSize,
            price: orderPrice,
            gridLevel: 0, // Unknown level
            placedAt: new Date(order.receivedTime || Date.now())
          });
          imported++;
          console.log(`[GridEngine] Imported orphan order: ${baseSymbol} BUY @ $${orderPrice} (no matching grid)`);
        }
      }

      console.log(`[GridEngine] Import complete: ${imported} imported, ${skipped} skipped`);
      console.log(`[GridEngine] Now tracking ${this.pendingOrders.size} pending orders`);

    } catch (error) {
      console.error('[GridEngine] Error importing existing orders:', error);
    }
  }

  private async syncWithKraken(): Promise<void> {
    if (this.pendingOrders.size === 0) return;

    try {
      // Get open orders from Kraken
      const openOrdersResponse = await this.apiRequest('GET', '/openorders');

      if (openOrdersResponse.result === 'success') {
        const krakenOpenOrders = new Set<string>();

        for (const order of openOrdersResponse.openOrders || []) {
          krakenOpenOrders.add(order.order_id);
        }

        // Check which of our pending orders are no longer in open orders = filled!
        for (const [orderId, pendingOrder] of this.pendingOrders.entries()) {
          if (!krakenOpenOrders.has(orderId)) {
            // Order is not in open orders - check if it was filled
            console.log(`[GridEngine] Order ${orderId} no longer pending, checking fills...`);
            await this.checkOrderFilled(orderId, pendingOrder);
          }
        }
      }

      // Also sync positions from Kraken
      await this.syncPositionsFromKraken();

    } catch (error) {
      console.error('[GridEngine] Sync with Kraken error:', error);
    }
  }

  private async checkOrderFilled(orderId: string, pendingOrder: { orderId: string; symbol: string; side: 'buy' | 'sell'; size: number; price: number; gridLevel: number; placedAt: Date; positionId?: string }): Promise<void> {
    try {
      // Get recent fills
      const fillsResponse = await this.apiRequest('GET', '/fills');

      if (fillsResponse.result === 'success') {
        const fills = fillsResponse.fills || [];

        // Find fills for this order
        const orderFills = fills.filter((f: any) => f.order_id === orderId);

        if (orderFills.length > 0) {
          // Order was filled!
          const totalFilled = orderFills.reduce((sum: number, f: any) => sum + f.size, 0);
          const avgPrice = orderFills.reduce((sum: number, f: any) => sum + f.price * f.size, 0) / totalFilled;

          console.log(`[GridEngine] ORDER FILLED! ${orderId}: ${pendingOrder.side} ${totalFilled} ${pendingOrder.symbol} @ ${avgPrice}`);

          const state = this.states.get(pendingOrder.symbol);
          if (state) {
            const symbolInfo = getSymbol(pendingOrder.symbol);
            const fee = symbolInfo ? avgPrice * totalFilled * symbolInfo.makerFee : 0;

            // Check if this is a TP order (has positionId) or entry order (has positionSide)
            const isTPOrder = (pendingOrder as any).positionId !== undefined;
            const positionSide = (pendingOrder as any).positionSide as 'LONG' | 'SHORT' | undefined;

            if (!isTPOrder && (pendingOrder.side === 'buy' || positionSide === 'SHORT')) {
              // Entry order filled - open new position
              // BUY = LONG entry, SELL with positionSide='SHORT' = SHORT entry
              const side = pendingOrder.side === 'buy' ? 'LONG' : 'SHORT';

              const grid = state.gridLevels.find(g => g.orderId === orderId);
              if (grid) {
                grid.orderStatus = 'filled';
                grid.filled = true;
              }

              // Calculate TP price based on position side
              const tpPrice = side === 'LONG'
                ? avgPrice * (1 + this.config.gridSpacing / 100)   // LONG: TP above
                : avgPrice * (1 - this.config.gridSpacing / 100);  // SHORT: TP below

              const position: GridPosition = {
                id: `P${++this.tradeIdCounter}`,
                symbol: pendingOrder.symbol,
                side,
                entryPrice: avgPrice,
                size: totalFilled,
                margin: this.config.positionPerGrid,
                gridLevel: pendingOrder.gridLevel,
                tpPrice,
                entryFee: fee,
                entryTime: pendingOrder.placedAt,
                krakenOrderId: orderId
              };

              state.positions.push(position);
              state.totalFees += fee;

              console.log(`[GridEngine] ${side} Position OPENED: ${pendingOrder.symbol} @ ${avgPrice}, size=${totalFilled}, TP=${tpPrice.toFixed(4)}`);
              this.emit('position_opened', position);

              // Now place TP order
              await this.placeTakeProfitOrder(position);

            } else if (isTPOrder) {
              // TP order filled - close position
              const position = state.positions.find(p => p.tpOrderId === orderId);
              if (position) {
                // PnL calculation depends on position side
                const isLong = position.side?.toLowerCase() === 'long';
                const pnl = isLong
                  ? (avgPrice - position.entryPrice) * position.size    // LONG: profit when exit > entry
                  : (position.entryPrice - avgPrice) * position.size;   // SHORT: profit when entry > exit
                const exitFee = fee;

                const trade: GridTrade = {
                  id: `T${++this.tradeIdCounter}`,
                  symbol: pendingOrder.symbol,
                  entryPrice: position.entryPrice,
                  exitPrice: avgPrice,
                  pnl,
                  fees: position.entryFee + exitFee,
                  netPnl: pnl - position.entryFee - exitFee,
                  entryTime: position.entryTime,
                  exitTime: new Date(),
                  result: 'TP'
                };

                state.trades.push(trade);
                state.realizedPnL += trade.netPnl;
                state.totalFees += exitFee;

                // Reset grid level so it can be used again
                const grid = state.gridLevels.find(g => g.level === position.gridLevel);
                if (grid) {
                  grid.filled = false;
                  grid.orderId = undefined;
                  grid.orderStatus = undefined;
                }

                // Remove position
                const idx = state.positions.indexOf(position);
                if (idx >= 0) state.positions.splice(idx, 1);

                console.log(`[GridEngine] Position CLOSED (TP): ${pendingOrder.symbol} @ ${avgPrice}, PnL=${trade.netPnl.toFixed(2)}`);
                this.emit('position_closed', trade);
              }
            }
          }

          // Remove from pending
          this.pendingOrders.delete(orderId);

        } else {
          // Order might have been cancelled
          const orderAge = Date.now() - pendingOrder.placedAt.getTime();
          if (orderAge > 60000) { // 1 minute old without fill = likely cancelled
            console.log(`[GridEngine] Order ${orderId} appears cancelled (no fills after 1 min)`);
            this.pendingOrders.delete(orderId);

            // Reset grid level for BUY orders only
            if (pendingOrder.side === 'buy') {
              const state = this.states.get(pendingOrder.symbol);
              if (state) {
                const grid = state.gridLevels.find(g => g.orderId === orderId);
                if (grid) {
                  grid.orderStatus = 'cancelled';
                  grid.orderId = undefined;
                  grid.filled = false;
                }
              }
            }
          }
        }
      }
    } catch (error) {
      console.error('[GridEngine] Check order filled error:', error);
    }
  }

  private async placeTakeProfitOrder(position: GridPosition): Promise<void> {
    const symbolInfo = getSymbol(position.symbol);
    if (!symbolInfo) return;

    try {
      // FIX: Use correct side based on position type
      // LONG positions close with SELL, SHORT positions close with BUY
      // Handle both uppercase (local) and lowercase (synced from Kraken)
      const isLong = position.side.toLowerCase() === 'long';
      const tpSide = isLong ? 'sell' : 'buy';
      const orderResult = await this.placeLimitOrder(position.symbol, tpSide, position.size, position.tpPrice);

      if (orderResult.success && orderResult.orderId) {
        position.tpOrderId = orderResult.orderId;

        // Track TP order as pending so we know when it fills
        this.pendingOrders.set(orderResult.orderId, {
          orderId: orderResult.orderId,
          symbol: position.symbol,
          side: tpSide,
          size: position.size,
          price: position.tpPrice,
          gridLevel: position.gridLevel,
          placedAt: new Date(),
          positionId: position.id // Link to the position this TP is for
        });

        console.log(`[GridEngine] TP order placed: ${tpSide.toUpperCase()} ${position.size} ${position.symbol} @ ${position.tpPrice} (${orderResult.orderId})`);
      } else {
        console.error(`[GridEngine] Failed to place TP order: ${orderResult.error}`);
      }
    } catch (error) {
      console.error('[GridEngine] TP order error:', error);
    }
  }

  private async syncPositionsFromKraken(): Promise<void> {
    try {
      const posResponse = await this.apiRequest('GET', '/openpositions');

      if (posResponse.result === 'success') {
        this.krakenPositions.clear();

        for (const pos of posResponse.openPositions || []) {
          // Convert Kraken symbol to our symbol
          const symbol = this.krakenSymbolToInternal(pos.symbol);
          if (symbol) {
            this.krakenPositions.set(pos.symbol, {
              symbol,
              side: pos.side,
              size: pos.size,
              price: pos.price,
              pnl: pos.unrealizedFunding || 0
            });

            // Also sync to state.positions for UI display
            const state = this.states.get(symbol);
            if (state) {
              // Check if position already exists in state
              const existingPos = state.positions.find(p =>
                p.symbol === symbol &&
                Math.abs(p.size - pos.size) < 0.001 &&
                Math.abs(p.entryPrice - pos.price) < 0.0001
              );

              if (!existingPos) {
                // Add position from Kraken to state
                const newPos: GridPosition = {
                  id: `K${++this.tradeIdCounter}`,
                  symbol: symbol,
                  entryPrice: pos.price,
                  size: pos.size,
                  margin: pos.size * pos.price / (this.config.leverage || 5),
                  gridLevel: 0,
                  tpPrice: pos.side === 'long'
                    ? pos.price * (1 + (this.config.gridSpacing || 0.5) / 100)
                    : pos.price * (1 - (this.config.gridSpacing || 0.5) / 100),
                  entryFee: 0,
                  entryTime: new Date(),
                  krakenOrderId: pos.symbol + '_synced',
                  side: pos.side as 'long' | 'short'
                };
                state.positions.push(newPos);
                console.log(`[GridEngine] Synced position from Kraken: ${pos.side} ${pos.size} ${symbol} @ ${pos.price}`);
              }

              // Update unrealized P&L
              const currentPrice = state.currentPrice || pos.price;
              state.positions.forEach(p => {
                if (p.symbol === symbol) {
                  const priceDiff = currentPrice - p.entryPrice;
                  p.unrealizedPnL = p.side === 'short'
                    ? -priceDiff * p.size
                    : priceDiff * p.size;
                }
              });
            }
          }
        }

        if (this.krakenPositions.size > 0) {
          console.log(`[GridEngine] Synced ${this.krakenPositions.size} positions from Kraken`);
        }
      }
    } catch (error) {
      console.error('[GridEngine] Sync positions error:', error);
    }
  }

  private krakenSymbolToInternal(krakenSymbol: string): string | null {
    // PF_XBTUSD -> BTC, PF_ETHUSD -> ETH
    if (krakenSymbol.startsWith('PF_') || krakenSymbol.startsWith('FI_')) {
      const base = krakenSymbol.slice(3, -3); // Remove prefix and USD suffix
      if (base === 'XBT') return 'BTC';
      return base;
    }
    return null;
  }

  // ============= AUTO STRATEGY =============

  private startStrategyAnalysis(): void {
    console.log('[GridEngine] Starting auto strategy analysis (every 4 hours)');

    // Analyze immediately on start
    setTimeout(() => this.analyzeAndAdaptStrategy(), 5000);

    // Then analyze every 4 hours (14400000 ms)
    this.strategyAnalysisInterval = setInterval(() => {
      this.analyzeAndAdaptStrategy();
    }, 4 * 60 * 60 * 1000);
  }

  private stopStrategyAnalysis(): void {
    if (this.strategyAnalysisInterval) {
      clearInterval(this.strategyAnalysisInterval);
      this.strategyAnalysisInterval = null;
      console.log('[GridEngine] Stopped auto strategy analysis');
    }
  }

  private analyzeAndAdaptStrategy(): void {
    // Need enough price history for analysis
    const anyHistory = Array.from(this.priceHistory.values()).find(h => h.length >= 50);
    if (!anyHistory) {
      console.log('[GridEngine] Not enough price history for strategy analysis, waiting...');
      return;
    }

    // Use BTC history as primary indicator (or first available symbol)
    const primarySymbol = this.config.symbols.includes('BTC') ? 'BTC' : this.config.symbols[0];
    const history = this.priceHistory.get(primarySymbol);

    if (!history || history.length < 50) {
      console.log(`[GridEngine] Not enough ${primarySymbol} history for analysis`);
      return;
    }

    // Convert price history to candle format for analysis
    // Since we only have close prices, we estimate OHLC
    const candles = this.convertHistoryToCandles(history);

    // Analyze market conditions
    const analysis = analyzeMarketConditions(candles);
    const newStrategy = analysis.recommendation;

    console.log(`[GridEngine] Market Analysis: condition=${analysis.condition}, trend=${analysis.trend}, volatility=${analysis.volatility}`);
    console.log(`[GridEngine] Recommendation: ${newStrategy} (confidence: ${analysis.confidence}%)`);
    console.log(`[GridEngine] Reason: ${analysis.reason}`);

    // Check if strategy should change
    if (newStrategy !== this.currentStrategy) {
      const oldStrategy = this.currentStrategy;
      this.currentStrategy = newStrategy;
      this.lastStrategyChange = Date.now();

      // Get new parameters
      const newParams = getStrategyParams(newStrategy, this.config.initialCapital);

      // Update config
      const oldConfig = { ...this.config };
      this.config.gridSpacing = newParams.gridSpacing;
      this.config.gridLevels = newParams.gridLevels;
      this.config.leverage = newParams.leverage;
      this.config.maxDrawdownPercent = newParams.maxDrawdownPercent;
      this.config.trendFilterEnabled = newParams.trendFilterEnabled;

      console.log(`[GridEngine] Strategy changed: ${oldStrategy || 'none'} -> ${newStrategy}`);
      console.log(`[GridEngine] New params: spacing=${newParams.gridSpacing}%, levels=${newParams.gridLevels}, leverage=${newParams.leverage}x`);

      // Rebuild grids with new spacing
      this.rebalanceGrids();

      // Emit event
      this.emit('strategy_changed', {
        from: oldStrategy,
        to: newStrategy,
        reason: analysis.reason,
        confidence: analysis.confidence,
        marketCondition: analysis.condition,
        newParams
      });
    } else {
      console.log(`[GridEngine] Strategy unchanged: ${this.currentStrategy}`);
    }
  }

  private convertHistoryToCandles(history: number[]): Array<{ open: number; high: number; low: number; close: number; time: number }> {
    const candles: Array<{ open: number; high: number; low: number; close: number; time: number }> = [];

    // Group prices into pseudo-candles (every 5 prices = 1 candle for analysis)
    const candleSize = 5;
    const now = Date.now();

    for (let i = 0; i < history.length; i += candleSize) {
      const slice = history.slice(i, i + candleSize);
      if (slice.length < 2) continue;

      const open = slice[0];
      const close = slice[slice.length - 1];
      const high = Math.max(...slice);
      const low = Math.min(...slice);

      candles.push({
        open,
        high,
        low,
        close,
        time: now - (history.length - i) * 60000 // Simulate 1-minute intervals
      });
    }

    return candles;
  }

  getCurrentStrategy(): StrategyPreset | null {
    return this.currentStrategy;
  }

  async closeAllPositions(): Promise<void> {
    for (const [symbol, state] of this.states) {
      for (const position of [...state.positions]) {
        await this.closePosition(symbol, position, 'CLOSE');
      }
    }
  }

  rebalanceGrids(): void {
    for (const [symbol, state] of this.states) {
      if (state.currentPrice > 0) {
        this.setupGrid(symbol, state.currentPrice);
      }
    }
    this.emit('grids_rebalanced');
  }

  pause(symbol?: string): void {
    if (symbol) {
      const state = this.states.get(symbol);
      if (state) {
        state.isPaused = true;
        state.pauseReason = 'Manual pause';
      }
    } else {
      for (const state of this.states.values()) {
        state.isPaused = true;
        state.pauseReason = 'Manual pause';
      }
    }
    this.emit('paused', { symbol });
  }

  resume(symbol?: string): void {
    if (symbol) {
      const state = this.states.get(symbol);
      if (state) {
        state.isPaused = false;
        state.pauseReason = null;
      }
    } else {
      for (const state of this.states.values()) {
        state.isPaused = false;
        state.pauseReason = null;
      }
    }
    this.emit('resumed', { symbol });
  }

  getStatus(): {
    isRunning: boolean;
    mode: string;
    uptime: number;
    symbols: string[];
    totalEquity: number;
    totalRealizedPnL: number;
    totalUnrealizedPnL: number;
    totalPositions: number;
    totalTrades: number;
    totalFees: number;
    krakenEquity: number;
    krakenAvailableMargin: number;
    autoStrategy: boolean;
    currentStrategy: string | null;
    lastStrategyChange: number | null;
  } {
    let totalRealizedPnL = 0;
    let totalUnrealizedPnL = 0;
    let totalPositions = 0;
    let totalTrades = 0;
    let totalFees = 0;

    for (const state of this.states.values()) {
      totalRealizedPnL += state.realizedPnL;
      totalUnrealizedPnL += state.unrealizedPnL;
      totalPositions += state.positions.length;
      totalTrades += state.trades.length;
      totalFees += state.totalFees;
    }

    // Calculate equity based on mode
    let displayEquity: number;
    let displayAvailableMargin: number;

    if (this.config.mode === 'test') {
      // Paper mode: equity = initial capital + realized P&L + unrealized P&L
      displayEquity = this.testBalance + totalUnrealizedPnL;
      displayAvailableMargin = this.testBalance;
    } else {
      // Demo/Live mode: use Kraken values
      displayEquity = this.krakenEquity;
      displayAvailableMargin = this.krakenAvailableMargin;
    }

    return {
      isRunning: this.isRunning,
      mode: this.config.mode,
      uptime: Math.floor((Date.now() - this.startTime) / 1000),
      symbols: this.config.symbols,
      totalEquity: displayEquity,
      totalRealizedPnL,
      totalUnrealizedPnL,
      totalPositions,
      totalTrades,
      totalFees,
      krakenEquity: displayEquity,
      krakenAvailableMargin: displayAvailableMargin,
      autoStrategy: this.config.autoStrategy || false,
      currentStrategy: this.currentStrategy,
      lastStrategyChange: this.lastStrategyChange || null
    };
  }

  getSymbolState(symbol: string): GridState | undefined {
    return this.states.get(symbol);
  }

  getAllStates(): Map<string, GridState> {
    return this.states;
  }

  getTickers(): Map<string, KrakenTicker> {
    return this.tickers;
  }

  getConfig(): GridEngineConfig {
    return { ...this.config };
  }

  updateConfig(newConfig: Partial<GridEngineConfig>): void {
    this.config = { ...this.config, ...newConfig };
    this.emit('config_updated', this.config);
  }
}

// ============= PRESETS =============

export const GRID_ENGINE_PRESETS = {
  conservative: {
    gridSpacing: 0.6,
    gridLevels: 8,
    positionPerGrid: 30,
    leverage: 3,
    maxOpenPositions: 6,
    maxDrawdownPercent: 25,
    trendFilterThreshold: 3,
    useLimitOrders: true
  },

  balanced: {
    gridSpacing: 0.5,
    gridLevels: 10,
    positionPerGrid: 50,
    leverage: 5,
    maxOpenPositions: 10,
    maxDrawdownPercent: 30,
    trendFilterThreshold: 3,
    useLimitOrders: true
  },

  aggressive: {
    gridSpacing: 0.4,
    gridLevels: 12,
    positionPerGrid: 80,
    leverage: 8,
    maxOpenPositions: 15,
    maxDrawdownPercent: 40,
    trendFilterThreshold: 5,
    useLimitOrders: false
  }
};

export default GridTradingEngine;
